package io.pythagoras.common.grayloglogger;

import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.AppenderBase;
import com.google.gson.Gson;
import io.pythagoras.common.grayloglogger.config.GraylogProperties;
import io.pythagoras.common.loggercommon.LogObject;
import org.graylog2.gelfclient.GelfConfiguration;
import org.graylog2.gelfclient.GelfMessageBuilder;
import org.graylog2.gelfclient.GelfMessageLevel;
import org.graylog2.gelfclient.GelfTransports;
import org.graylog2.gelfclient.transport.GelfTransport;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.net.InetSocketAddress;
import java.util.Arrays;
import java.util.UUID;

@Service
public class GraylogAppender extends AppenderBase<ILoggingEvent> {

    private GelfTransport transport;

    private GraylogProperties properties;

    private String appName;

    private String instanceId = UUID.randomUUID().toString();

    @Autowired
    public void setProperties(GraylogProperties properties) {
        this.properties = properties;
    }

    @Value("${spring.application.name:NAME_NOT_FOUND}")
    public void setAppName(String appName) {
        this.appName = appName;
    }

    @Override
    public void start() {
        this.configure();
        super.start();
    }

    @Override
    protected void append(ILoggingEvent eventObject) {
        Gson gson = new Gson();

        GelfMessageBuilder message = new GelfMessageBuilder(eventObject.getFormattedMessage(), properties.getHost());
        message.level(convertLevels(eventObject.getLevel()))
                .additionalField("app_name", appName)
                .additionalField("timestamp", eventObject.getTimeStamp())
                .additionalField("class",eventObject.getLoggerName())
                .additionalField("thread_name",eventObject.getThreadName())
                .additionalField("instance_id",instanceId);

        Object[] args = eventObject.getArgumentArray();

        if (args != null && args.length > 0) {

            Object obj = args[0];

            if (obj instanceof LogObject) {
                LogObject lobj = (LogObject) obj;
                message.additionalField("statusCode", lobj.statusCode)
                        .additionalField("stackTrace", gson.toJson(lobj.exception.getStackTrace()));
                if(args.length > 1) {
                        message.additionalField("objects", gson.toJson(Arrays.copyOfRange(args, 1, args.length - 1)));
                }
            } else {
                message.additionalField("objects", gson.toJson(args));
            }
        } else if (eventObject.getThrowableProxy() != null) {
            message.additionalField("stackTrace", gson.toJson(eventObject.getThrowableProxy().getStackTraceElementProxyArray()));
        }

        Boolean sendResult = this.transport.trySend(message.build());
        if (!sendResult) {
            //TODO: Handle a full logging buffer somehow.
            System.out.println("ERROR: Graylog buffer full.");
        }


    }

    private void configure() {
        GelfConfiguration config = new GelfConfiguration(new InetSocketAddress(properties.getHost(), properties.getPort()))
                .transport(GelfTransports.UDP)
                .queueSize(512)
                .connectTimeout(5000)
                .reconnectDelay(1000)
                .tcpNoDelay(true)
                .sendBufferSize(32768);

        this.transport = GelfTransports.create(config);
    }

    private GelfMessageLevel convertLevels(Level inLevel) {
        switch (inLevel.toInt()) {
            case Level.ALL_INT:
            case Level.OFF_INT:
                return GelfMessageLevel.DEBUG;
            case Level.DEBUG_INT:
            case Level.TRACE_INT:
                return GelfMessageLevel.DEBUG;
            case Level.ERROR_INT:
                return GelfMessageLevel.ERROR;
            case Level.INFO_INT:
                return GelfMessageLevel.INFO;
            case Level.WARN_INT:
                return GelfMessageLevel.WARNING;
        }
        throw new RuntimeException("Unable to convert Log Level");
    }

}
