package io.proofdock.chaos.middleware.core.loader.proofdock;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.proofdock.chaos.middleware.core.AppConfig;
import io.proofdock.chaos.middleware.core.Chaos;
import io.proofdock.chaos.middleware.core.MessageHandler;
import io.proofdock.chaos.middleware.core.model.AttackActionSchema;
import java.io.UnsupportedEncodingException;
import java.util.Arrays;
import java.util.List;
import org.apache.http.HttpResponse;
import org.apache.http.client.fluent.Request;
import org.apache.http.client.fluent.Response;
import org.apache.http.entity.StringEntity;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ProofdockAttackLoaderTask implements Runnable {

  private static Logger log = LoggerFactory.getLogger(ProofdockAttackLoaderTask.class);


  private static final String PROOFDOCK_API_DEFAULT_URL = "https://chaosapi.proofdock.io";
  private static final String PROOFDOCK_API_DEFAULT_ROUTE = "/v1/attacks/synchronize";

  private final ObjectMapper mapper;
  private final AppConfig appConfig;

  public ProofdockAttackLoaderTask(AppConfig appConfig) {
    this.appConfig = appConfig;
    this.mapper = new ObjectMapper();
  }

  @Override
  public void run() {
    try {
      // Arrange
      StringEntity entity = createRequestBody(this.appConfig);
      String token = this.appConfig.get(AppConfig.PROOFDOCK_API_TOKEN, "");
      String targetUri = this.appConfig
          .get(AppConfig.PROOFDOCK_API_URL, ProofdockAttackLoaderTask.PROOFDOCK_API_DEFAULT_URL);

      // Fetch
      Response response = Request.Post(targetUri + ProofdockAttackLoaderTask.PROOFDOCK_API_DEFAULT_ROUTE)
          .addHeader("Authorization", "Bearer " + token)
          .body(entity)
          .connectTimeout(13000)
          .socketTimeout(13000)
          .execute();

      // Act
      HttpResponse httpResponse = response.returnResponse();
      int statusCode = httpResponse.getStatusLine().getStatusCode();
      if (statusCode >= 200 && statusCode < 400) {
        String content = EntityUtils.toString(httpResponse.getEntity());
        List<AttackActionSchema> actionSchemas = Arrays.asList(
            this.mapper.readValue(content, AttackActionSchema[].class));
        Chaos.getInstance().loadAttacks(actionSchemas);
      } else {
        throw new Exception(MessageHandler.get("attackloader.load.throwexception") + " Reason: Invalid status code returned from api loader: " + statusCode);
      }
    } catch (Exception e) {
      log.error(MessageHandler.get("proofdockattackloadertask.run.exception"), e);
      Chaos.getInstance().unloadAttacks();
    }
  }

  private StringEntity createRequestBody(AppConfig appConfig) throws UnsupportedEncodingException {
    String id = appConfig.get(AppConfig.APPLICATION_ID, "");
    String env = appConfig.get(AppConfig.APPLICATION_ENV, "");
    String name = appConfig.get(AppConfig.APPLICATION_NAME, "");
    String json = String.format("{\"id\":\"%s\",\"env\":\"%s\",\"name\":\"%s\"}", id, env, name);
    StringEntity result = new StringEntity(json);

    return result;
  }
}
