package io.proofdock.chaos.middleware.core.loader;

import io.proofdock.chaos.middleware.core.AppConfig;
import io.proofdock.chaos.middleware.core.MessageHandler;
import io.proofdock.chaos.middleware.core.loader.proofdock.ProofdockAttackLoader;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class is a base class for implementing different type of attack config loaders.
 * <p>
 * For example, different loaders implementation can load an attack configuration from a file, network or other source.
 * The loading mechanism depends on the configured settings.
 */
public abstract class AttackLoader {

  private static Logger log = LoggerFactory.getLogger(AttackLoader.class);
  private static final String DEFAULT_LOADER = "proofdock";

  public AttackLoader(AppConfig appConfig) {
    this.appConfig = appConfig;
  }

  protected AppConfig appConfig;
  private ScheduledExecutorService scheduledThreadPool = Executors.newScheduledThreadPool(1);

  /**
   * Whether the loader is allowed to call the endpoint, e.g. when all parameters are set.
   *
   * @return true if allowed to call
   */
  protected abstract boolean isAllowedToCallEndpoint();

  /**
   * Load attack actions from provider.
   * 
   * @throws Exception unable to load attacks
   */
  public void load() throws Exception {
    if (isAllowedToCallEndpoint()) {
      Runnable task = createTask();
      scheduledThreadPool.scheduleAtFixedRate(task, 0, 30, TimeUnit.SECONDS);
    } else {
      throw new Exception(MessageHandler.get("attackloader.load.throwexception"));
    }
  }

  /**
   * Load function. Call callback function to set new attack actions.
   * 
   * @return task for loading configuration 
   */
  protected abstract Runnable createTask();

  /**
   * Load the attack configuration from the API provider
   *
   * @param app_config the application configuration
   * @return The attack loader strategy to be returned
   */
  public static AttackLoader get(AppConfig app_config) {
    String provider = app_config.get(AppConfig.ATTACK_LOADER, DEFAULT_LOADER);

    AttackLoader loader = null;
    if (provider.equals(DEFAULT_LOADER)) {
      loader = new ProofdockAttackLoader(app_config);
    } else {
      log.warn(String.format(MessageHandler.get("attackloader.get.noproperloader"), provider));
    }

    if (loader !=null && !loader.isAllowedToCallEndpoint()) {
      log.info(String.format(MessageHandler.get("attackloader.get.noconfig"), provider));
      loader = null;
    }

    return loader;
  }

}
