package io.proofdock.chaos.middleware.core;

import io.proofdock.chaos.middleware.core.error.ChaosMiddlewareError;
import io.proofdock.chaos.middleware.core.loader.AttackLoader;
import io.proofdock.chaos.middleware.core.model.AttackActionSchema;
import io.proofdock.chaos.middleware.core.model.AttackContext;
import io.proofdock.chaos.middleware.core.model.AttackSchema;
import io.proofdock.chaos.middleware.core.model.AttackTargetSchema;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Chaos {

  private static Logger log = LoggerFactory.getLogger(Chaos.class);

  public static final String ATTACK_ACTION_FAULT = "fault";
  public static final String ATTACK_ACTION_DELAY = "delay";

  private static Chaos instance;

  private AppConfig config;
  private AttackLoader attackLoader;
  private List<AttackActionSchema> loadedActions;
  private Inject inject;

  private Chaos() {
    inject = new Inject();
    loadedActions = new ArrayList<>();
  }

  /**
   * Get a Chaos instance.
   *
   * @return a Chaos instance (singleton)
   */
  public static Chaos getInstance() {
    if (instance == null) {
      instance = new Chaos();
    }
    return instance;
  }

  /**
   * Register the application and start the attacks loader.
   *
   * @param config application configuration
   * 
   * @throws Exception unable to register chaos middleware
   */
  public void register(AppConfig config) throws Exception {
    assert config != null;

    if (this.config == null) {
      this.config = config;
      this.attackLoader = AttackLoader.get(config);
      this.attackLoader.load();
    }
  }

  /**
   * Reset the application configuration.
   */
  public void deregister() {
    this.config = null;
  }

  /**
   * Unload the application's loaded attacks.
   */
  public void unloadAttacks() {
    this.loadedActions = new ArrayList<>();
  }

  /**
   * Set the application's attacks.
   * 
   * @param actionSchemas load attacks
   */
  public void loadAttacks(List<AttackActionSchema> actionSchemas) {
    this.loadedActions = actionSchemas;
  }

  /**
   * Attack the application.
   *
   * @param attackInput   information about attack, if null loaded attacks will be
   *                      considered
   * @param attackContext context of the attack (e.g url)
   *
   * @throws Exception throws exception if fault attack was performed
   */
  public void attack(AttackSchema attackInput, AttackContext attackContext) throws Exception {
    try {
      // Attack via request header (from client)
      if (attackInput != null) {
        execute_attacks(attackInput.getTarget(), attackInput.getActions(), attackContext);
      } else if (loadedActions.size() > 0) {
        execute_attacks(null, this.loadedActions, attackContext);
      }
    } catch (ChaosMiddlewareError cme) {
      this.log.debug(MessageHandler.get("chaos.attack.chaosmiddlewareerror"));
      if (cme.getCause() != null) {
        throw (Exception) cme.getCause();
      } else {
        throw cme;
      }
    } catch (Exception e) {
      this.log.error(MessageHandler.get("chaos.attack.exception"), e);
    }
  }

  private void execute_attacks(AttackTargetSchema target, List<AttackActionSchema> actions, AttackContext attackContext)
      throws Exception {
    for (AttackActionSchema action : actions) {

      if (!isAppTargeted(target)) {
        continue;
      }

      if (!isRouteTargeted(attackContext.getRoute(), action.getRoute())) {
        continue;
      }

      if (!isLuckyToBeAttacked(action.getProbability())) {
        continue;
      }

      if (action.getName().equals(Chaos.ATTACK_ACTION_DELAY)) {
        inject.delay(action.getValue());
      }

      if (action.getName().equals(Chaos.ATTACK_ACTION_FAULT)) {
        inject.fault(action.getValue());
      }
    }
  }

  private boolean isLuckyToBeAttacked(String probability) {
    boolean is_lucky = Dice.roll(probability);

    return is_lucky;
  }

  private boolean isRouteTargeted(String attackCtxRoute, String actionRoute) {
    if (StringUtil.isBlankOrEmpty(attackCtxRoute) || StringUtil.isBlankOrEmpty(actionRoute)) {
      return true;
    }

    String text = actionRoute.replace("/*", "/[\\w-]*");
    Pattern pattern = Pattern.compile(text);
    Matcher matcher = pattern.matcher(attackCtxRoute);

    boolean result = matcher.find();
    return result;
  }

  private boolean isAppTargeted(AttackTargetSchema target) {
    if (target == null) {
      return true;
    }

    String application = target.getApplication();
    String environment = target.getEnvironment();

    boolean isAppTargeted = StringUtil.isBlankOrEmpty(application) || !StringUtil.isBlankOrEmpty(application)
        && application.equals(this.config.get(AppConfig.APPLICATION_NAME, ""));

    boolean isEnvTargeted = StringUtil.isBlankOrEmpty(environment) || !StringUtil.isBlankOrEmpty(environment)
        && environment.equals(this.config.get(AppConfig.APPLICATION_ENV, ""));

    return isAppTargeted && isEnvTargeted;
  }

}
