package io.polyglotted.elastic.client;

import io.polyglotted.common.config.SettingsHolder;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.experimental.Accessors;
import org.apache.http.Header;
import org.apache.http.message.BasicHeader;

import java.net.URI;

import static java.nio.charset.StandardCharsets.UTF_8;
import static org.apache.commons.codec.binary.Base64.encodeBase64String;
import static org.apache.http.HttpHeaders.AUTHORIZATION;

@SuppressWarnings("unused")
@NoArgsConstructor @Accessors(chain = true) @Getter @Setter
public class ElasticSettings {
    String scheme = "https";
    String host = "localhost";
    int port = 9200;
    int connectTimeoutMillis = 5000;
    int socketTimeoutMillis = 300_000;
    boolean insecureSsl = false;
    boolean predeterminedSsl = true;
    boolean enableSniffer = false;
    BootstrapAuth bootstrap = new BootstrapAuth();

    public ElasticSettings setBootstrap(String user, String password) { return setBootstrap(new BootstrapAuth(user, password)); }

    Header bootstrapAuth() { return bootstrap.bootstrapAuth(); }

    public static ElasticSettings from(SettingsHolder holder, String prefix) {
        return elasticSettings()
            .setScheme(holder.stringValue(prefix + ".scheme", "https"))
            .setHost(holder.stringValue(prefix + ".host", "localhost"))
            .setPort(holder.intValue(prefix + ".port", 9200))
            .setConnectTimeoutMillis(holder.intValue(prefix + ".connectTimeoutMillis", 5000))
            .setSocketTimeoutMillis(holder.intValue(prefix + ".socketTimeoutMillis", 300_000))
            .setInsecureSsl(holder.booleanValue(prefix + ".insecureSsl", true))
            .setPredeterminedSsl(holder.booleanValue(prefix + ".predeterminedSsl", true))
            .setEnableSniffer(holder.booleanValue(prefix + ".enableSniffer", false))
            .setBootstrap(holder.stringValue(prefix + ".bootstrap.username", "admin"),
                holder.stringValue(prefix + ".bootstrap.password", null));
    }

    public static ElasticSettings from(String url, boolean insecureSsl, boolean predeterminedSsl) {
        URI uri = URI.create(url);
        return elasticSettings().setScheme(uri.getScheme()).setHost(uri.getHost()).setPort(uri.getPort())
            .setInsecureSsl(insecureSsl).setPredeterminedSsl(predeterminedSsl);
    }

    public static ElasticSettings elasticSettings() { return new ElasticSettings(); }

    @NoArgsConstructor @AllArgsConstructor @Accessors(chain = true) @Getter @Setter
    public static class BootstrapAuth {
        String username = "elastic";
        String password = null;

        Header bootstrapAuth() {
            return (username == null || password == null) ? null :
                new BasicHeader(AUTHORIZATION, encodeBase64String((username + ":" + password).getBytes(UTF_8)));
        }
    }
}