package io.polyglotted.elastic.client;

import io.polyglotted.common.model.MapResult;
import io.polyglotted.common.util.ListBuilder.SimpleListBuilder;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.Header;
import org.apache.http.HttpHost;
import org.apache.http.util.EntityUtils;
import org.elasticsearch.client.Node;
import org.elasticsearch.client.Request;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.Response;
import org.elasticsearch.client.RestClient;
import org.elasticsearch.client.sniff.NodesSniffer;
import org.elasticsearch.client.sniff.Sniffer;

import java.util.List;
import java.util.Map;

import static io.polyglotted.common.util.BaseSerializer.deserialize;
import static io.polyglotted.common.util.CollUtil.transformList;
import static io.polyglotted.common.util.ListBuilder.simpleListBuilder;
import static io.polyglotted.common.util.MapRetriever.MAP_CLASS;
import static java.util.Objects.requireNonNull;
import static org.apache.http.HttpStatus.SC_MULTIPLE_CHOICES;
import static org.apache.http.HttpStatus.SC_OK;

@Slf4j @RequiredArgsConstructor
public final class InternalHostsSniffer implements NodesSniffer {
    private final RestClient lowLevelClient;
    private final ElasticSettings settings;
    private final Header bootstrapAuth;

    static Sniffer buildSniffer(RestClient lowLevelClient, ElasticSettings settings, Header authHeader) {
        return Sniffer.builder(lowLevelClient).setNodesSniffer(new InternalHostsSniffer(lowLevelClient, settings, authHeader)).build();
    }

    @Override public List<Node> sniff() {
        MapResult result = deserialize(performCliRequest());
        SimpleListBuilder<String> addresses = simpleListBuilder();
        for (Map.Entry<String, Object> entry : result.mapVal("nodes").entrySet()) {
            addresses.add((String) MAP_CLASS.cast(entry.getValue()).get("ip"));
        }
        return transformList(addresses.build(), node -> new Node(new HttpHost(requireNonNull(node), settings.getPort(), settings.getScheme())));
    }

    private String performCliRequest() {
        try {
            Request request = new Request("GET", "/_nodes/http");
            if (bootstrapAuth != null) { addHeader(request, bootstrapAuth); }
            Response response = lowLevelClient.performRequest(request);

            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode >= SC_OK && statusCode < SC_MULTIPLE_CHOICES) { return EntityUtils.toString(response.getEntity()); }
            log.warn("sniff failed: " + response.getStatusLine().getReasonPhrase());

        } catch (Exception ex) { log.error("sniff error", ex); }
        return "{}";
    }

    static Request addHeader(Request request, Header header) {
        RequestOptions.Builder options = request.getOptions().toBuilder();
        options.setHttpAsyncResponseConsumerFactory(RequestOptions.DEFAULT.getHttpAsyncResponseConsumerFactory());
        if (header != null) { options.addHeader(header.getName(), header.getValue()); }
        request.setOptions(options);
        return request;
    }
}