package io.polyglotted.elastic.client;

import io.polyglotted.common.model.AuthHeader;
import io.polyglotted.common.model.MapResult;
import io.polyglotted.common.util.HttpRequestBuilder.HttpReqType;
import io.polyglotted.common.util.MapBuilder.ImmutableMapBuilder;
import lombok.Getter;
import lombok.SneakyThrows;
import lombok.experimental.Accessors;
import org.apache.http.ConnectionClosedException;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.entity.StringEntity;
import org.apache.http.util.EntityUtils;
import org.elasticsearch.client.Response;
import org.elasticsearch.client.ResponseException;
import org.elasticsearch.client.RestClient;
import org.elasticsearch.client.RestClientBuilder;
import org.elasticsearch.client.sniff.Sniffer;

import javax.annotation.Nullable;
import java.io.IOException;
import java.net.ConnectException;
import java.util.List;
import java.util.Map;

import static io.polyglotted.common.model.MapResult.immutableResult;
import static io.polyglotted.common.util.BaseSerializer.deserialize;
import static io.polyglotted.common.util.BaseSerializer.deserializeToList;
import static io.polyglotted.common.util.HttpRequestBuilder.HttpReqType.DELETE;
import static io.polyglotted.common.util.HttpRequestBuilder.HttpReqType.GET;
import static io.polyglotted.common.util.HttpRequestBuilder.HttpReqType.POST;
import static io.polyglotted.common.util.HttpRequestBuilder.HttpReqType.PUT;
import static io.polyglotted.common.util.MapBuilder.immutableMapBuilder;
import static io.polyglotted.common.util.MapRetriever.asMap;
import static io.polyglotted.common.util.MapRetriever.deepRetrieve;
import static io.polyglotted.common.util.MapRetriever.mapVal;
import static io.polyglotted.common.util.MapRetriever.reqdStr;
import static io.polyglotted.common.util.StrUtil.notNullOrEmpty;
import static io.polyglotted.common.util.ThreadUtil.safeSleep;
import static io.polyglotted.elastic.client.ElasticException.checkState;
import static io.polyglotted.elastic.client.ElasticException.throwEx;
import static io.polyglotted.elastic.client.InternalHostsSniffer.buildSniffer;
import static java.util.Collections.emptyMap;
import static org.apache.http.HttpStatus.SC_MULTIPLE_CHOICES;
import static org.apache.http.HttpStatus.SC_OK;
import static org.apache.http.entity.ContentType.APPLICATION_JSON;

@Accessors(fluent = true)
public class ElasticRestClient implements ElasticClient {
    private final RestClient internalClient;
    private final Sniffer sniffer;
    @Nullable @Getter private final AuthHeader bootstrapAuth;

    ElasticRestClient(RestClientBuilder builder, ElasticSettings settings, @Nullable AuthHeader bootstrapAuth) {
        internalClient = builder.build();
        this.sniffer = settings.enableSniffer ? buildSniffer(this.internalClient, settings, bootstrapAuth) : null;
        this.bootstrapAuth = bootstrapAuth;
    }

    @Override @SneakyThrows public void close() { if (sniffer != null) { sniffer.close(); } internalClient.close(); }

    @SuppressWarnings("ALL")
    @Override public ElasticClient waitForStatus(AuthHeader auth, String status) {
        try {
            for (int i = 0; i <= 300; i++) {
                performCliRequest(auth, GET, "/_cluster/health?wait_for_status=" + status); break;
            }
        } catch (ConnectException | ConnectionClosedException retry) {
            safeSleep(1000); waitForStatus(auth, status);
        } catch (Exception ioe) { throw throwEx("waitForStatus failed", ioe); }
        return this;
    }

    @Override public MapResult clusterHealth(AuthHeader auth) { return deserialize(simpleGet(auth, "/_cluster/health", "clusterHealth")); }

    @Override public boolean indexExists(AuthHeader auth, String index) {
        try {
            return internalClient.performRequest("HEAD", "/" + index, headers(auth))
                .getStatusLine().getStatusCode() == SC_OK;
        } catch (Exception ioe) { throw throwEx("indexExists failed", ioe); }
    }

    @Override public MapResult indexNameFor(AuthHeader auth, String alias) {
        ImmutableMapBuilder<String, String> result = immutableMapBuilder();
        try {
            List<Map<String, Object>> list = deserializeToList(performCliRequest(auth, GET, "/_cat/aliases"
                + (notNullOrEmpty(alias) ? "/" + alias : "") + "?h=index,alias&format=json"));
            for (Map<String, Object> map : list) { result.put(reqdStr(map, "alias"), reqdStr(map, "index")); }
            return result.result();
        } catch (Exception ioe) { throw throwEx("indexNameFor failed", ioe); }
    }

    @Override public String createIndex(AuthHeader auth, String index, String body) {
        try {
            MapResult result = deserialize(simplePut(auth, "/" + index, body, "createIndex"));
            checkState(result.boolVal("acknowledged", false) && result.boolVal("shards_acknowledged", false), "unable to create index");
            return result.reqdStr("index");
        } catch (Exception ioe) { throw throwEx("createIndex failed", ioe); }
    }

    @Override public void dropIndex(AuthHeader auth, String index) { simpleDelete(auth, "/" + index, "dropIndex"); }

    @Override public void forceRefresh(AuthHeader auth, String index) {
        try {
            performCliRequest(auth, POST, "/" + index + "/_refresh");
        } catch (Exception ioe) { throw throwEx("forceRefresh failed", ioe); }
    }

    @Override public void putSettings(AuthHeader auth, String index, String settingsJson) {
        try {
            performCliRequest(auth, POST, "/" + index + "/_close");
            simplePut(auth, "/" + index + "/_settings", settingsJson, "putSettings");
            performCliRequest(auth, POST, "/" + index + "/_open");

        } catch (Exception e) { throw throwEx("putSettings failed", e); }
    }

    @Override public MapResult getSettings(AuthHeader auth, String index) {
        try {
            MapResult result = deserialize(simpleGet(auth, "/" + index + "/_settings", "getSettings"));
            return immutableResult(deepRetrieve(asMap(result.first()), "settings.index"));
        } catch (Exception e) { throw throwEx("getSettings failed", e); }
    }

    @Override public void putMapping(AuthHeader auth, String index, String mappingJson) {
        simplePut(auth, "/" + index + "/_mapping/_doc", mappingJson, "putMapping");
    }

    @Override public MapResult getMapping(AuthHeader auth, String index) {
        try {
            MapResult result = deserialize(performCliRequest(auth, GET, "/" + index + "/_mapping/_doc"));
            return immutableResult(mapVal(asMap(result.first()), "mappings"));
        } catch (Exception e) { throw throwEx("getMapping failed", e); }
    }

    @Override public void putPipeline(AuthHeader auth, String id, String body) { simplePut(auth, "/_ingest/pipeline/" + id, body, "putPipeline"); }

    @Override public boolean pipelineExists(AuthHeader auth, String id) { return simpleGet(auth, "/_ingest/pipeline/" + id, "pipelineExists") != null; }

    @Override public void deletePipeline(AuthHeader auth, String id) { simpleDelete(auth, "/_ingest/pipeline/" + id, "deletePipeline"); }

    @Override public void putTemplate(AuthHeader auth, String name, String body) { simplePut(auth, "/_template/" + name, body, "putTemplate"); }

    @Override public boolean templateExists(AuthHeader auth, String name) { return simpleGet(auth, "/_template/" + name, "templateExists") != null; }

    @Override public void deleteTemplate(AuthHeader auth, String name) { simpleDelete(auth, "/_template/" + name, "deleteTemplate"); }

    @Override public String simpleGet(AuthHeader auth, String endpoint, String methodName) {
        Exception throwable;
        try {
            return performCliRequest(auth, GET, endpoint);

        } catch (ResponseException re) {
            if (re.getResponse().getStatusLine().getStatusCode() == 404) { return null; }
            throwable = re;
        } catch (Exception ioe) { throwable = ioe; }
        throw throwEx(methodName + " failed", throwable);
    }

    @Override public String simplePut(AuthHeader auth, String endpoint, String body, String methodName) {
        try {
            return performCliRequest(PUT, endpoint, emptyMap(), new StringEntity(body, APPLICATION_JSON), headers(auth));
        } catch (Exception ioe) { throw throwEx(methodName + " failed", ioe); }
    }

    @Override public void simpleDelete(AuthHeader auth, String endpoint, String methodName) {
        try {
            performCliRequest(auth, DELETE, endpoint);
        } catch (Exception ioe) { throw throwEx(methodName + " failed", ioe); }
    }

    @Override public String performCliRequest(AuthHeader auth, HttpReqType method, String endpoint) throws IOException {
        return performCliRequest(method, endpoint, emptyMap(), null, headers(auth));
    }

    @Override public String performCliRequest(HttpReqType method, String endpoint, Map<String, String> params,
                                              HttpEntity entity, Header... headers) throws IOException {
        Response response = internalClient.performRequest(method.name(), endpoint, params, entity, headers);
        int statusCode = response.getStatusLine().getStatusCode();
        checkState(statusCode >= SC_OK && statusCode < SC_MULTIPLE_CHOICES, response.getStatusLine().getReasonPhrase());
        return EntityUtils.toString(response.getEntity());
    }

    private Header[] headers(AuthHeader authHeader) {
        return authHeader != null ? authHeader.headers() : (bootstrapAuth != null ? bootstrapAuth.headers() : new Header[0]);
    }
}