/*
 * Decompiled with CFR 0.152.
 */
package io.polyglotted.elastic.discovery;

import com.amazonaws.AmazonClientException;
import com.amazonaws.http.IdleConnectionReaper;
import com.amazonaws.services.ec2.AmazonEC2;
import com.amazonaws.services.ec2.model.DescribeInstancesRequest;
import com.amazonaws.services.ec2.model.DescribeInstancesResult;
import com.amazonaws.services.ec2.model.Filter;
import com.amazonaws.services.ec2.model.GroupIdentifier;
import com.amazonaws.services.ec2.model.Instance;
import com.amazonaws.services.ec2.model.Reservation;
import com.amazonaws.services.ec2.model.Tag;
import io.polyglotted.elastic.discovery.Ec2Service;
import io.polyglotted.elastic.discovery.Ec2ServiceImpl;
import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.elasticsearch.common.settings.Settings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class UnicastHostsProvider
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(UnicastHostsProvider.class);
    private final AmazonEC2 client;
    private final boolean bindAnyGroup;
    private final Set<String> groups;
    private final Map<String, List<String>> tags;
    private final Set<String> availabilityZones;
    private final String hostType;

    public static List<String> fetchEc2Addresses(Settings settings) throws IOException {
        try (UnicastHostsProvider provider = new UnicastHostsProvider(settings);){
            List<String> list = provider.fetchAddresses();
            return list;
        }
    }

    private UnicastHostsProvider(Settings settings) {
        this.client = Ec2ServiceImpl.client(settings);
        this.hostType = (String)Ec2Service.HOST_TYPE_SETTING.get(settings);
        this.bindAnyGroup = (Boolean)Ec2Service.ANY_GROUP_SETTING.get(settings);
        this.groups = new HashSet<String>();
        this.groups.addAll((Collection)Ec2Service.GROUPS_SETTING.get(settings));
        this.tags = Ec2Service.TAG_SETTING.getAsMap(settings);
        this.availabilityZones = new HashSet<String>();
        this.availabilityZones.addAll((Collection)Ec2Service.AVAILABILITY_ZONES_SETTING.get(settings));
        if (log.isDebugEnabled()) {
            log.debug("using host_type [{}], tags [{}], groups [{}] with any_group [{}], availability_zones [{}]", new Object[]{this.hostType, this.tags, this.groups, this.bindAnyGroup, this.availabilityZones});
        }
    }

    @Override
    public void close() throws IOException {
        if (this.client != null) {
            this.client.shutdown();
        }
        IdleConnectionReaper.shutdown();
    }

    private List<String> fetchAddresses() {
        DescribeInstancesResult descInstances;
        ArrayList<String> addresses = new ArrayList<String>();
        try {
            descInstances = this.client.describeInstances(this.buildDescribeInstancesRequest());
        }
        catch (AmazonClientException e) {
            log.info("Exception while retrieving instance list from AWS API: {}", (Object)e.getMessage());
            log.debug("Full exception:", (Throwable)e);
            return addresses;
        }
        log.trace("building dynamic unicast discovery nodes...");
        for (Reservation reservation : descInstances.getReservations()) {
            for (Instance instance : reservation.getInstances()) {
                if (!this.groups.isEmpty()) {
                    List instanceSecurityGroups = instance.getSecurityGroups();
                    ArrayList<String> securityGroupNames = new ArrayList<String>(instanceSecurityGroups.size());
                    ArrayList<String> securityGroupIds = new ArrayList<String>(instanceSecurityGroups.size());
                    for (GroupIdentifier sg : instanceSecurityGroups) {
                        securityGroupNames.add(sg.getGroupName());
                        securityGroupIds.add(sg.getGroupId());
                    }
                    if (this.bindAnyGroup) {
                        if (Collections.disjoint(securityGroupNames, this.groups) && Collections.disjoint(securityGroupIds, this.groups)) {
                            log.trace("filtering out instance {} based on groups {}, not part of {}", new Object[]{instance.getInstanceId(), instanceSecurityGroups, this.groups});
                            continue;
                        }
                    } else if (!securityGroupNames.containsAll(this.groups) && !securityGroupIds.containsAll(this.groups)) {
                        log.trace("filtering out instance {} based on groups {}, does not include all of {}", new Object[]{instance.getInstanceId(), instanceSecurityGroups, this.groups});
                        continue;
                    }
                }
                String address = null;
                if (this.hostType.equals("private_dns")) {
                    address = instance.getPrivateDnsName();
                } else if (this.hostType.equals("private_ip")) {
                    address = instance.getPrivateIpAddress();
                } else if (this.hostType.equals("public_dns")) {
                    address = instance.getPublicDnsName();
                } else if (this.hostType.equals("public_ip")) {
                    address = instance.getPublicIpAddress();
                } else if (this.hostType.startsWith("tag:")) {
                    String tagName = this.hostType.substring("tag:".length());
                    log.debug("reading hostname from [{}] instance tag", (Object)tagName);
                    List tags = instance.getTags();
                    for (Tag tag : tags) {
                        if (!tag.getKey().equals(tagName)) continue;
                        address = tag.getValue();
                        log.debug("using [{}] as the instance address", (Object)address);
                    }
                } else {
                    throw new IllegalArgumentException(this.hostType + " is unknown for discovery.ec2.host_type");
                }
                if (address == null) continue;
                addresses.add(address);
            }
        }
        log.debug("using dynamic discovery addresses {}", addresses);
        return addresses;
    }

    private DescribeInstancesRequest buildDescribeInstancesRequest() {
        DescribeInstancesRequest describeInstancesRequest = new DescribeInstancesRequest().withFilters(new Filter[]{new Filter("instance-state-name").withValues(new String[]{"running", "pending"})});
        for (Map.Entry<String, List<String>> tagFilter : this.tags.entrySet()) {
            describeInstancesRequest.withFilters(new Filter[]{new Filter("tag:" + tagFilter.getKey()).withValues((Collection)tagFilter.getValue())});
        }
        if (!this.availabilityZones.isEmpty()) {
            describeInstancesRequest.withFilters(new Filter[]{new Filter("availability-zone").withValues(this.availabilityZones)});
        }
        return describeInstancesRequest;
    }
}

