/**
 * Copyright 2010 The PlayN Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package playn.core;

import react.Slot;
import react.Function;

/**
 * Defines and dispatches keyboard events. Three events are generated by keyboard input:
 * <ul>
 * <li> When any key is depressed, a {@link KeyEvent} is emitted indicating the logical key that
 * was depressed. </li>
 * <li> If the depressed key also corresponds to a printable character ('c' for example, but not
 * shift or alt), a {@link TypedEvent} is emitted to inform the app of the typed character. The
 * typed character will account for whether the shift key is depressed and will be appropriately
 * mapped to the uppercase equivalent or the appropriate alternate character (for example, # for 3,
 * in the US keyboard layout). The typed event is delivered immediately after the pressed event.
 * </li>
 * <li> When a key is released, a {@link KeyEvent} is emitted, indicating the logical key that was
 * released. </li>
 * </ul>
 */
public abstract class Keyboard {

  /** The base class for all keyboard events. */
  public static class Event extends playn.core.Event.Input {
    protected Event (int flags, double time) {
      super(flags, time);
    }
  }

  /** An event dispatched for key press/release. */
  public static class KeyEvent extends Event {
    /** The key that triggered this event, e.g. {@link Key#A}, etc. */
    public final Key key;
    /** Whether the key is down or up. */
    public final boolean down;

    public KeyEvent (int flags, double time, Key key, boolean down) {
      super(flags, time);
      this.key = key;
      this.down = down;
    }

    @Override protected String name() {
      return "Key";
    }
    @Override protected void addFields(StringBuilder builder) {
      super.addFields(builder);
      builder.append(", key=").append(key).append(", down=").append(down);
    }
  }

  /** An event dispatched when a printable character is typed. */
  public static class TypedEvent extends Event {
    /** The character typed to trigger this event, e.g. 'c'. */
    public char typedChar;

    public TypedEvent (int flags, double time, char typedChar) {
      super(flags, time);
      this.typedChar = typedChar;
    }

    @Override protected String name () {
      return "Typed";
    }
    @Override protected void addFields(StringBuilder builder) {
      super.addFields(builder);
      builder.append(", typedChar=").append(typedChar);
    }
  }

  /** A slot which only dispatches on {@link KeyEvent}s. */
  public static abstract class KeySlot extends Slot<Event> {
    public void onEmit (Event event) {
      if (event instanceof KeyEvent) onEmit((KeyEvent)event);
    }
    public abstract void onEmit (KeyEvent event);
  }

  /** A slot which only dispatches on {@link TypedEvent}s. */
  public static abstract class TypedSlot extends Slot<Event> {
    public void onEmit (Event event) {
      if (event instanceof TypedEvent) onEmit((TypedEvent)event);
    }
    public abstract void onEmit (TypedEvent event);
  }

  /** Enumerates the different available mobile keyboard types. See {@link Input#getText}. */
  public static enum TextType { DEFAULT, NUMBER, EMAIL, URL; }

  /**
   * A function which collects only {@code KeyEvent} events. Use it to obtain only key events like
   * so:
   *
   * <pre>{@code
   * Input.keyboardEvents.collect(Keyboard.isKeyEvent).connect(event -> {
   *   // handle key events here (event has type KeyEvent)
   * });
   * }</pre>
   */
  public static Function<Event, KeyEvent> isKeyEvent = new Function<Event, KeyEvent>() {
    public KeyEvent apply (Event event) {
      return (event instanceof KeyEvent) ? (KeyEvent)event : null;
    }
  };

  /**
   * A function which collects only {@code KeyEvent} events. Use it to obtain only key events like
   * so:
   *
   * <pre>{@code
   * Input.keyboardEvents.collect(Keyboard.isTypedEvent).connect(event -> {
   *   // handle typed events here (event has type TypedEvent)
   * });
   * }</pre>
   */
  public static Function<Event, KeyEvent> isTypedEvent = new Function<Event, KeyEvent>() {
    public KeyEvent apply (Event event) {
      return (event instanceof KeyEvent) ? (KeyEvent)event : null;
    }
  };

  /**
   * Returns a collector function for key events for {@code key}. Use it to obtain only events for
   * a particular key like so:
   *
   * <pre>{@code
   * Input.keyboardEvents.collect(Keyboard.isKey(Key.X)).connect(event -> {
   *   // handle the 'x' key being pressed or released
   * });
   * }</pre>
   */
  public static Function<Event, KeyEvent> isKey (final Key key) {
    return new Function<Event, KeyEvent>() {
      public KeyEvent apply (Event event) {
        if (event instanceof KeyEvent && ((KeyEvent)event).key == key) {
          return (KeyEvent)event;
        } else {
          return null;
        }
      }
    };
  }
}
