package io.pdfire.client

import java.time.Duration
import java.time.OffsetDateTime
import java.net.URL
import java.io.InputStream
import java.io.BufferedInputStream
import java.io.IOException

/**
 * Represents a conversion that has failed or is served via a CDN / custom cloud storage.
 *
 * @property initializedAt Initilization time of the conversion.
 * @property finishedAt Completion / Failure time of the conversion.
 * @property status Either "succeeded" or "failed".
 * @property error Error message if the conversion failed.
 * @property result The conversion result.
 */
class Conversion constructor(
    val initializedAt: OffsetDateTime,
    val finishedAt: OffsetDateTime,
    val status: String,
    val error: String?,
    val result: ConversionResult
): AbstractResult() {
    companion object {}

    private var pdf: ByteArray? = null

    override fun getBytes() = download()

    /**
     * Download the PDF from the CDN or custom cloud storage.
     */
    fun download(): ByteArray {
        if (status != "succeeded") {
            throw Exception("Cannot download failed conversion.")
        }

        if (pdf != null) {
            return pdf!!
        }

        val stream = BufferedInputStream(URL(result.url).openStream())

        stream.use { s ->
            pdf = s.readAllBytes()
        }

        return pdf!!
    }
}

/**
 * The conversion result.
 *
 * @property size Size of the PDF in bytes.
 * @property width Width of the PDF in pixel.
 * @property height Height of the PDF in pixel.
 * @property expiresAt Expiration date of the PDF (if served via the PDFire.io CDN).
 * @property runtime Runtime of the conversion in milliseconds.
 * @property url URL to the PDF.
 */
data class ConversionResult(val size: Int, val width: Int, val height: Int, val expiresAt: OffsetDateTime?, val runtime: Duration, val url: String)
