package io.pdfire.client

/**
 * Configuration for a conversion of multiple documents (merge).
 *
 * Options may require a minimum subscription tier to be usable.
 * See the <a href="https://docs.pdfire.io/options/#cdn">documentation</a> for information on the available options.
 *
 * @property documents Conversion options ([ConversionParameters]) for the documents.
 * @property ownerPassword Secures the PDF with an owner password.
 * @property userPassword Secures the PDF with a user password.
 * @property cdn Indicates if the PDF should be served via the CDN.
 * @property storage Storage configuration for the PDF.
 */
class MergeParameters constructor(
    var documents: MutableList<ConversionParameters> = mutableListOf(),
    var ownerPassword: String? = null,
    var userPassword: String? = null,
    override var cdn: Boolean? = null,
    override var storage: StorageParameter? = null
): Parameters {
    /**
     * Serve the PDF via a custom cloud storage disk.
     */
    fun useStorage(configure: ((StorageParameter) -> Unit)?) {
        val param = StorageParameter()

        if (configure != null) {
            configure(param)
        }

        storage = param
    }

    /**
     * Serve the PDF via the default custom cloud storage disk.
     */
    fun useStorage(use: Boolean = true) {
        if (use) {
            useStorage { _ -> }
        } else {
            storage = null
        }
    }

    /**
     * Add documents to the PDF.
     */
    fun addDocuments(vararg documents: ConversionParameters) {
        this.documents.addAll(documents)
    }

    /**
     * Add a document to the PDF.
     */
    fun addDocument(configure: ((ConversionParameters) -> Unit)? = null) {
        val doc = ConversionParameters()

        if (configure != null) {
            configure(doc)
        }

        documents.add(doc)
    }

    /**
     * Create a map from the parameters.
     */
    override fun toMap(): Map<String, Any?> = mapOf(
        Pair("documents", documents.map { it.toMap() }),
        Pair("ownerPassword", ownerPassword),
        Pair("userPassword", userPassword),
        Pair("cdn", cdn),
        Pair("storage", storage?.toMap())
    ).filter { it.value != null }
}