package io.pdfire.client

import java.time.Duration

/**
 * Configuration for a conversion.
 *
 * Options may require a minimum subscription tier to be usable.
 * See the <a href="https://docs.pdfire.io/options/#cdn">documentation</a> for information on the available options.
 *
 * @property html The HTML string to convert to a PDF.
 * @property url The URL to convert to a PDF.
 * @property cdn Indicates if the PDF should be served via the CDN.
 * @property storage Storage configuration for the PDF.
 * @property landscape Indicates if the page should be captured in landscape mode.
 * @property printBackground Indicates if the background images of the page should be included in the PDF.
 * @property scale Sets the zoom of the page.
 * @property format Paper format.
 * @property paperWidth Paper width.
 * @property paperHeight Paper height.
 * @property margin Margins of the page.
 * @property marginTop Top margin of the page (overrides [margin]).
 * @property marginRight Right margin of the page (overrides [margin]).
 * @property marginBottom Bottom margin of the page (overrides [margin]).
 * @property marginLeft Left margin of the page (overrides [margin]).
 * @property pageRanges Page ranges that should be captured.
 * @property headerTemplate HTML template for the PDF header.
 * @property footerTemplate HTML template for the PDF footer.
 * @property preferCSSPageSize Indicates if the page's CSS rule should take precedence
 * over the [paperWidth], [paperHeight] and [format] options.
 * @property viewportWidth Width of the browser's viewport.
 * @property viewportHeight Height of the browser's viewport.
 * @property blockAds Indicates if the browser should try to remove ads from the page.
 * @property selector DOM selector for the HTML element that should be captured.
 * @property waitForSelector DOM selector the service should wait for before capturing the page.
 * @property waitForSelectorTimeout Timeout for the [waitForSelector] option.
 * @property waitUntil Browser event to wait for before capturing the page.
 * @property waitUntilTimeout Timeout for the [waitUntil] option.
 * @property delay Delay the capturing of the page for this duration.
 * @property timeout General timeout for the conversion request.
 * @property headers Custom HTTP headers that will be attached to the request
 * from the PDFire service to the provided URL.
 * @property emulateMedia Which CSS media to emulate.
 * @property ownerPassword Secures the PDF with an owner password.
 * @property userPassword Secures the PDF with a user password.
 * @property allowErrorPage Allow HTTP status codes outside the 2xx range to be captured.
 * @property optimize Apply CSS optimizations to the page.
 */
class ConversionParameters constructor(
    var html: String? = null,
    var url: String? = null,
    override var cdn: Boolean? = null,
    override var storage: StorageParameter? = null,
    var landscape: Boolean? = null,
    var printBackground: Boolean? = null,
    var scale: Float? = null,
    var format: PaperFormat? = null,
    var paperWidth: String? = null,
    var paperHeight: String? = null,
    var margin: String? = null,
    var marginTop: String? = null,
    var marginRight: String? = null,
    var marginBottom: String? = null,
    var marginLeft: String? = null,
    var pageRanges: MutableList<String> = mutableListOf(),
    var headerTemplate: String? = null,
    var footerTemplate: String? = null,
    var preferCSSPageSize: Boolean? = null,
    var viewportWidth: Float? = null,
    var viewportHeight: Float? = null,
    var blockAds: Boolean? = null,
    var selector: String? = null,
    var waitForSelector: String? = null,
    var waitForSelectorTimeout: Duration? = null,
    var waitUntil: WaitUntil? = null,
    var waitUntilTimeout: Duration? = null,
    var delay: Duration? = null,
    var timeout: Duration? = null,
    var headers: MutableMap<String, Any> = mutableMapOf(),
    var emulateMedia: Media? = null,
    var ownerPassword: String? = null,
    var userPassword: String? = null,
    var allowErrorPage: Boolean? = null,
    var optimize: Boolean? = null
): Parameters {
    /**
     * Secure the PDF with an owner / user password.
     */
    fun secure(owner: String? = null, user: String? = null) {
        ownerPassword = owner
        userPassword = user
    }

    /**
     * Set the paper width of the PDF in pixel.
     */
    fun setPaperWidth(width: Int) {
        paperWidth = width.toString()
    }

    /**
     * Set the paper width of the PDF in pixel.
     */
    fun setPaperWidthPixel(width: Int) {
        paperWidth = width.toString() + "px"
    }

    /**
     * Set the paper width of the PDF in inches.
     */
    fun setPaperWidthInch(width: Float) {
        paperWidth = width.toString() + "in"
    }

    /**
     * Set the paper width of the PDF in millimeter.
     */
    fun setPaperWidthMillimeter(width: Float) {
        paperWidth = width.toString() + "mm"
    }

    /**
     * Set the paper width of the PDF in centimeter.
     */
    fun setPaperWidthCentimeter(width: Float) {
        paperWidth = width.toString() + "cm"
    }

    /**
     * Set the paper height of the PDF in pixel.
     */
    fun setPaperHeight(height: Int) {
        paperHeight = height.toString()
    }

    /**
     * Set the paper height of the PDF in pixel.
     */
    fun setPaperHeightPixel(height: Float) {
        paperHeight = height.toString() + "px"
    }

    /**
     * Set the paper height of the PDF in inches.
     */
    fun setPaperHeightInch(height: Float) {
        paperHeight = height.toString() + "in"
    }

    /**
     * Set the paper height of the PDF in millimeter.
     */
    fun setPaperHeightMillimeter(height: Float) {
        paperHeight = height.toString() + "mm"
    }

    /**
     * Set the paper height of the PDF in centimeter.
     */
    fun setPaperHeightCentimeter(height: Float) {
        paperHeight = height.toString() + "cm"
    }

    /**
     * Set the margins of the PDF in pixel.
     */
    fun setMargin(margin: Int) {
        this.margin = margin.toString()
    }

    /**
     * Set the margins of the PDF in pixel.
     */
    fun setMarginPixel(margin: Float) {
        this.margin = margin.toString() + "px"
    }

    /**
     * Set the margins of the PDF in inches.
     */
    fun setMarginInch(margin: Float) {
        this.margin = margin.toString() + "in"
    }

    /**
     * Set the margins of the PDF in millimeter.
     */
    fun setMarginMillimeter(margin: Float) {
        this.margin = margin.toString() + "mm"
    }

    /**
     * Set the margins of the PDF in centimeter.
     */
    fun setMarginCentimeter(margin: Float) {
        this.margin = margin.toString() + "cm"
    }

    /**
     * Set the top margin of the PDF in pixel.
     */
    fun setMarginTop(margin: Int) {
        marginTop = margin.toString()
    }

    /**
     * Set the top margin of the PDF in pixel.
     */
    fun setMarginTopPixel(margin: Float) {
        marginTop = margin.toString() + "px"
    }

    /**
     * Set the top margin of the PDF in inches.
     */
    fun setMarginTopInch(margin: Float) {
        marginTop = margin.toString() + "in"
    }

    /**
     * Set the top margin of the PDF in millimeter.
     */
    fun setMarginTopMillimeter(margin: Float) {
        marginTop = margin.toString() + "mm"
    }

    /**
     * Set the top margin of the PDF in centimeter.
     */
    fun setMarginTopCentimeter(margin: Float) {
        marginTop = margin.toString() + "cm"
    }

    /**
     * Set the right margin of the PDF in pixel.
     */
    fun setMarginRight(margin: Int) {
        marginRight = margin.toString()
    }

    /**
     * Set the right margin of the PDF in pixel.
     */
    fun setMarginRightPixel(margin: Float) {
        marginRight = margin.toString() + "px"
    }

    /**
     * Set the right margin of the PDF in inches.
     */
    fun setMarginRightInch(margin: Float) {
        marginRight = margin.toString() + "in"
    }

    /**
     * Set the right margin of the PDF in millimeter.
     */
    fun setMarginRightMillimeter(margin: Float) {
        marginRight = margin.toString() + "mm"
    }

    /**
     * Set the right margin of the PDF in centimeter.
     */
    fun setMarginRightCentimeter(margin: Float) {
        marginRight = margin.toString() + "cm"
    }

    /**
     * Set the bottom margin of the PDF in pixel.
     */
    fun setMarginBottom(margin: Int) {
        marginBottom = margin.toString()
    }

    /**
     * Set the bottom margin of the PDF in pixel.
     */
    fun setMarginBottomPixel(margin: Float) {
        marginBottom = margin.toString() + "px"
    }

    /**
     * Set the bottom margin of the PDF in inches.
     */
    fun setMarginBottomInch(margin: Float) {
        marginBottom = margin.toString() + "in"
    }

    /**
     * Set the bottom margin of the PDF in millimeter.
     */
    fun setMarginBottomMillimeter(margin: Float) {
        marginBottom = margin.toString() + "mm"
    }

    /**
     * Set the bottom margin of the PDF in centimeter.
     */
    fun setMarginBottomCentimeter(margin: Float) {
        marginBottom = margin.toString() + "cm"
    }

    /**
     * Set the left margin of the PDF in pixel.
     */
    fun setMarginLeft(margin: Int) {
        marginLeft = margin.toString()
    }

    /**
     * Set the left margin of the PDF in pixel.
     */
    fun setMarginLeftPixel(margin: Float) {
        marginLeft = margin.toString() + "px"
    }

    /**
     * Set the left margin of the PDF in inches.
     */
    fun setMarginLeftInch(margin: Float) {
        marginLeft = margin.toString() + "in"
    }

    /**
     * Set the left margin of the PDF in millimeter.
     */
    fun setMarginLeftMillimeter(margin: Float) {
        marginLeft = margin.toString() + "mm"
    }

    /**
     * Set the left margin of the PDF in centimeter.
     */
    fun setMarginLeftCentimeter(margin: Float) {
        marginLeft = margin.toString() + "cm"
    }

    /**
     * Wait for the browser's "load" event before capturing the page.
     */
    fun waitUntilLoad() {
        waitUntil = WaitUntil.Load
    }

    /**
     * Wait for the browser's "DOMContentLoaded" event before capturing the page.
     */
    fun waitUntilDOM() {
        waitUntil = WaitUntil.DOM
    }

    /**
     * Serve the PDF via a custom cloud storage disk.
     */
    fun useStorage(configure: ((StorageParameter) -> Unit)?) {
        val param = StorageParameter()

        if (configure != null) {
            configure(param)
        }

        storage = param
    }

    /**
     * Serve the PDF via the default custom cloud storage disk.
     */
    fun useStorage(use: Boolean = true) {
        if (use) {
            useStorage { _ -> }
        } else {
            storage = null
        }
    }

    /**
     * Add custom [headers] to the request the PDFire server makes to the provided [url].
     */
    fun withHeaders(headers: Map<String, Any>) {
        this.headers.putAll(headers)
    }

    /**
     * Add page ranges that should be captured.
     */
    fun addPageRange(range: String) {
        pageRanges.add(range)
    }

    /**
     * Create a map from the parameters.
     */
    override fun toMap(): Map<String, Any?> = mapOf(
        Pair("html", html),
        Pair("url", url),
        Pair("cdn", cdn),
        Pair("landscape", landscape),
        Pair("printBackground", printBackground),
        Pair("scale", scale),
        Pair("format", format?.toString()),
        Pair("paperWidth", paperWidth),
        Pair("paperHeight", paperHeight),
        Pair("margin", margin),
        Pair("marginTop", marginTop),
        Pair("marginRight", marginRight),
        Pair("marginBottom", marginBottom),
        Pair("marginLeft", marginLeft),
        Pair("pageRanges", if (pageRanges.isNotEmpty()) pageRanges.joinToString(",") else null),
        Pair("headerTemplate", headerTemplate),
        Pair("footerTemplate", footerTemplate),
        Pair("preferCSSPageSize", preferCSSPageSize),
        Pair("viewportWidth", viewportWidth),
        Pair("viewportHeight", viewportHeight),
        Pair("blockAds", blockAds),
        Pair("selector", selector),
        Pair("waitForSelector", waitForSelector),
        Pair("waitForSelectorTimeout", waitForSelectorTimeout?.toMillis()),
        Pair("waitUntil", waitUntil.toString()),
        Pair("waitUntilTimeout", waitUntilTimeout?.toMillis()),
        Pair("delay", delay?.toMillis()),
        Pair("timeout", timeout?.toMillis()),
        Pair("headers", headers),
        Pair("emulateMedia", emulateMedia.toString()),
        Pair("ownerPassword", ownerPassword),
        Pair("userPassword", userPassword),
        Pair("allowErrorPage", allowErrorPage),
        Pair("optimize", optimize)
    ).filter { it.value != null }
}
