package io.pdfire.client

import java.time.Duration

/**
 * Configuration for a conversion.
 *
 * Options may require a minimum subscription tier to be usable.
 * See the <a href="https://docs.pdfire.io/options/#cdn">documentation</a> for information on the available options.
 *
 * @property html The HTML string to convert to a PDF.
 * @property url The URL to convert to a PDF.
 * @property cdn Indicates if the PDF should be served via the CDN.
 * @property storage Storage configuration for the PDF.
 * @property landscape Indicates if the page should be captured in landscape mode.
 * @property printBackground Indicates if the background images of the page should be included in the PDF.
 * @property scale Sets the zoom of the page.
 * @property format Paper format.
 * @property paperWidth Paper width.
 * @property paperHeight Paper height.
 * @property margin Margins of the page.
 * @property marginTop Top margin of the page (overrides [margin]).
 * @property marginRight Right margin of the page (overrides [margin]).
 * @property marginBottom Bottom margin of the page (overrides [margin]).
 * @property marginLeft Left margin of the page (overrides [margin]).
 * @property pageRanges Page ranges that should be captured.
 * @property headerTemplate HTML template for the PDF header.
 * @property footerTemplate HTML template for the PDF footer.
 * @property preferCSSPageSize Indicates if the page's CSS rule should take precedence
 * over the [paperWidth], [paperHeight] and [format] options.
 * @property viewportWidth Width of the browser's viewport.
 * @property viewportHeight Height of the browser's viewport.
 * @property blockAds Indicates if the browser should try to remove ads from the page.
 * @property selector DOM selector for the HTML element that should be captured.
 * @property waitForSelector DOM selector the service should wait for before capturing the page.
 * @property waitForSelectorTimeout Timeout for the [waitForSelector] option.
 * @property waitUntil Browser event to wait for before capturing the page.
 * @property waitUntilTimeout Timeout for the [waitUntil] option.
 * @property delay Delay the capturing of the page for this duration.
 * @property timeout General timeout for the conversion request.
 * @property headers Custom HTTP headers that will be attached to the request
 * from the PDFire service to the provided URL.
 * @property emulateMedia Which CSS media to emulate.
 * @property ownerPassword Secures the PDF with an owner password.
 * @property userPassword Secures the PDF with a user password.
 * @property allowErrorPage Allow HTTP status codes outside the 2xx range to be captured.
 * @property optimize Apply CSS optimizations to the page.
 */
class ConversionParameters constructor(
    var html: String? = null,
    var url: String? = null,
    var cdn: Boolean? = null,
    var storage: StorageParameter? = null,
    var landscape: Boolean? = null,
    var printBackground: Boolean? = null,
    var scale: Float? = null,
    var format: PaperFormat? = null,
    var paperWidth: String? = null,
    var paperHeight: String? = null,
    var margin: String? = null,
    var marginTop: String? = null,
    var marginRight: String? = null,
    var marginBottom: String? = null,
    var marginLeft: String? = null,
    var pageRanges: MutableList<String> = mutableListOf(),
    var headerTemplate: String? = null,
    var footerTemplate: String? = null,
    var preferCSSPageSize: Boolean? = null,
    var viewportWidth: Float? = null,
    var viewportHeight: Float? = null,
    var blockAds: Boolean? = null,
    var selector: String? = null,
    var waitForSelector: String? = null,
    var waitForSelectorTimeout: Duration? = null,
    var waitUntil: WaitUntil? = null,
    var waitUntilTimeout: Duration? = null,
    var delay: Duration? = null,
    var timeout: Duration? = null,
    var headers: MutableMap<String, Any> = mutableMapOf(),
    var emulateMedia: Media? = null,
    var ownerPassword: String? = null,
    var userPassword: String? = null,
    var allowErrorPage: Boolean? = null,
    var optimize: Boolean? = null
): Parameters {
    /**
     * Set the [html] option.
     */
    fun setHTML(html: String): ConversionParameters {
        this.html = html
        return this
    }

    /**
     * Set the [url] option.
     */
    fun setURL(url: String): ConversionParameters {
        this.url = url
        return this
    }

    /**
     * Secure the PDF with an owner / user password.
     */
    fun secure(owner: String? = null, user: String? = null): ConversionParameters {
        ownerPassword = owner
        userPassword = user
        return this
    }

    /**
     * Set the paper width of the PDF in pixel.
     */
    fun setPaperWidth(width: Int): ConversionParameters {
        paperWidth = width.toString()
        return this
    }

    /**
     * Set the paper width of the PDF. Accepts CSS units (px / in / mm / cm).
     */
    fun setPaperWidth(width: String): ConversionParameters {
        paperWidth = width
        return this
    }

    /**
     * Set the paper width of the PDF in pixel.
     */
    fun setPaperWidthPixel(width: Int): ConversionParameters {
        return setPaperWidth(width.toString() + "px")
    }

    /**
     * Set the paper width of the PDF in inches.
     */
    fun setPaperWidthInch(width: Float): ConversionParameters {
        return setPaperWidth(width.toString() + "in")
    }

    /**
     * Set the paper width of the PDF in millimeter.
     */
    fun setPaperWidthMillimeter(width: Float): ConversionParameters {
        return setPaperWidth(width.toString() + "mm")
    }

    /**
     * Set the paper width of the PDF in centimeter.
     */
    fun setPaperWidthCentimeter(width: Float): ConversionParameters {
        return setPaperWidth(width.toString() + "cm")
    }

    /**
     * Set the paper height of the PDF in pixel.
     */
    fun setPaperHeight(height: Int): ConversionParameters {
        paperHeight = height.toString()
        return this
    }

    /**
     * Set the paper height of the PDF. Accepts CSS units (px / in / mm / cm).
     */
    fun setPaperHeight(height: String): ConversionParameters {
        paperHeight = height
        return this
    }

    /**
     * Set the paper height of the PDF in pixel.
     */
    fun setPaperHeightPixel(height: Float): ConversionParameters {
        return setPaperHeight(height.toString() + "px")
    }

    /**
     * Set the paper height of the PDF in inches.
     */
    fun setPaperHeightInch(height: Float): ConversionParameters {
        return setPaperHeight(height.toString() + "in")
    }

    /**
     * Set the paper height of the PDF in millimeter.
     */
    fun setPaperHeightMillimeter(height: Float): ConversionParameters {
        return setPaperHeight(height.toString() + "mm")
    }

    /**
     * Set the paper height of the PDF in centimeter.
     */
    fun setPaperHeightCentimeter(height: Float): ConversionParameters {
        return setPaperHeight(height.toString() + "cm")
    }

    /**
     * Set the margins of the PDF in pixel.
     */
    fun setMargin(margin: Int): ConversionParameters {
        this.margin = margin.toString()
        return this
    }

    /**
     * Set the margins of the PDF. Accepts CSS units (px / in / mm / cm).
     */
    fun setMargin(margin: String): ConversionParameters {
        this.margin = margin
        return this
    }

    /**
     * Set the margins of the PDF in pixel.
     */
    fun setMarginPixel(margin: Float): ConversionParameters {
        return setMargin(margin.toString() + "px")
    }

    /**
     * Set the margins of the PDF in inches.
     */
    fun setMarginInch(margin: Float): ConversionParameters {
        return setMargin(margin.toString() + "in")
    }

    /**
     * Set the margins of the PDF in millimeter.
     */
    fun setMarginMillimeter(margin: Float): ConversionParameters {
        return setMargin(margin.toString() + "mm")
    }

    /**
     * Set the margins of the PDF in centimeter.
     */
    fun setMarginCentimeter(margin: Float): ConversionParameters {
        return setMargin(margin.toString() + "cm")
    }

    /**
     * Set the top margin of the PDF in pixel.
     */
    fun setMarginTop(margin: Int): ConversionParameters {
        marginTop = margin.toString()
        return this
    }

    /**
     * Set the top margin of the PDF. Accepts CSS units (px / in / mm / cm).
     */
    fun setMarginTop(margin: String): ConversionParameters {
        marginTop = margin
        return this
    }

    /**
     * Set the top margin of the PDF in pixel.
     */
    fun setMarginTopPixel(margin: Float): ConversionParameters {
        return setMarginTop(margin.toString() + "px")
    }

    /**
     * Set the top margin of the PDF in inches.
     */
    fun setMarginTopInch(margin: Float): ConversionParameters {
        return setMarginTop(margin.toString() + "in")
    }

    /**
     * Set the top margin of the PDF in millimeter.
     */
    fun setMarginTopMillimeter(margin: Float): ConversionParameters {
        return setMarginTop(margin.toString() + "mm")
    }

    /**
     * Set the top margin of the PDF in centimeter.
     */
    fun setMarginTopCentimeter(margin: Float): ConversionParameters {
        return setMarginTop(margin.toString() + "cm")
    }

    /**
     * Set the right margin of the PDF in pixel.
     */
    fun setMarginRight(margin: Int): ConversionParameters {
        marginRight = margin.toString()
        return this
    }

    /**
     * Set the right margin of the PDF. Accepts CSS units (px / in / mm / cm).
     */
    fun setMarginRight(margin: String): ConversionParameters {
        marginRight = margin
        return this
    }

    /**
     * Set the right margin of the PDF in pixel.
     */
    fun setMarginRightPixel(margin: Float): ConversionParameters {
        return setMarginRight(margin.toString() + "px")
    }

    /**
     * Set the right margin of the PDF in inches.
     */
    fun setMarginRightInch(margin: Float): ConversionParameters {
        return setMarginRight(margin.toString() + "in")
    }

    /**
     * Set the right margin of the PDF in millimeter.
     */
    fun setMarginRightMillimeter(margin: Float): ConversionParameters {
        return setMarginRight(margin.toString() + "mm")
    }

    /**
     * Set the right margin of the PDF in centimeter.
     */
    fun setMarginRightCentimeter(margin: Float): ConversionParameters {
        return setMarginRight(margin.toString() + "cm")
    }

    /**
     * Set the bottom margin of the PDF in pixel.
     */
    fun setMarginBottom(margin: Int): ConversionParameters {
        marginBottom = margin.toString()
        return this
    }

    /**
     * Set the bottom margin of the PDF. Accepts CSS units (px / in / mm / cm).
     */
    fun setMarginBottom(margin: String): ConversionParameters {
        marginBottom = margin
        return this
    }

    /**
     * Set the bottom margin of the PDF in pixel.
     */
    fun setMarginBottomPixel(margin: Float): ConversionParameters {
        return setMarginBottom(margin.toString() + "px")
    }

    /**
     * Set the bottom margin of the PDF in inches.
     */
    fun setMarginBottomInch(margin: Float): ConversionParameters {
        return setMarginBottom(margin.toString() + "in")
    }

    /**
     * Set the bottom margin of the PDF in millimeter.
     */
    fun setMarginBottomMillimeter(margin: Float): ConversionParameters {
        return setMarginBottom(margin.toString() + "mm")
    }

    /**
     * Set the bottom margin of the PDF in centimeter.
     */
    fun setMarginBottomCentimeter(margin: Float): ConversionParameters {
        return setMarginBottom(margin.toString() + "cm")
    }

    /**
     * Set the left margin of the PDF in pixel.
     */
    fun setMarginLeft(margin: Int): ConversionParameters {
        marginLeft = margin.toString()
        return this
    }

    /**
     * Set the left margin of the PDF in pixel. Accepts CSS units (px / in / mm / cm).
     */
    fun setMarginLeft(margin: String): ConversionParameters {
        marginLeft = margin
        return this
    }

    /**
     * Set the left margin of the PDF in pixel.
     */
    fun setMarginLeftPixel(margin: Float): ConversionParameters {
        return setMarginLeft(margin.toString() + "px")
    }

    /**
     * Set the left margin of the PDF in inches.
     */
    fun setMarginLeftInch(margin: Float): ConversionParameters {
        return setMarginLeft(margin.toString() + "in")
    }

    /**
     * Set the left margin of the PDF in millimeter.
     */
    fun setMarginLeftMillimeter(margin: Float): ConversionParameters {
        return setMarginLeft(margin.toString() + "mm")
    }

    /**
     * Set the left margin of the PDF in centimeter.
     */
    fun setMarginLeftCentimeter(margin: Float): ConversionParameters {
        return setMarginLeft(margin.toString() + "cm")
    }

    /**
     * Wait for the browser's "load" event before capturing the page.
     */
    fun waitUntilLoad(): ConversionParameters {
        waitUntil = WaitUntil.Load
        return this
    }

    /**
     * Wait for the browser's "DOMContentLoaded" event before capturing the page.
     */
    fun waitUntilDOM(): ConversionParameters {
        waitUntil = WaitUntil.DOM
        return this
    }

    /**
     * Set the CDN option.
     */
    override fun setCDN(cdn: Boolean): ConversionParameters {
        this.cdn = cdn
        return this
    }

    /**
     * Serve the PDF via the PDFire.io CDN.
     */
    fun useCDN(): ConversionParameters {
        return setCDN(true)
    }

    /**
     * Capture the page in landscape mode.
     */
    fun setLandscape(landscape: Boolean = true): ConversionParameters {
        this.landscape = landscape
        return this
    }

    /**
     * Set the [printBackground] option.
     */
    fun setPrintBackground(printBackground: Boolean = true): ConversionParameters {
        this.printBackground = printBackground
        return this
    }

    /**
     * Set the [preferCSSPageSize] option.
     */
    fun setPreferCSSPageSize(preferCSSPageSize: Boolean = true): ConversionParameters {
        this.preferCSSPageSize = preferCSSPageSize
        return this
    }

    /**
     * Set the [blockAds] option.
     */
    fun setBlockAds(blockAds: Boolean = true): ConversionParameters {
        this.blockAds = blockAds
        return this
    }

    /**
     * Set the [allowErrorPage] option.
     */
    fun setAllowErrorPage(allowErrorPage: Boolean = true): ConversionParameters {
        this.allowErrorPage = allowErrorPage
        return this
    }

    /**
     * Set the [optimize] option.
     */
    fun setOptimize(optimize: Boolean = true): ConversionParameters {
        this.optimize = optimize
        return this
    }

    /**
     * Serve the PDF via a custom cloud storage disk.
     */
    override fun useStorage(configure: ((StorageParameter) -> Unit)?): ConversionParameters {
        val param = StorageParameter()

        if (configure != null) {
            configure(param)
        }

        storage = param
        return this
    }

    /**
     * Serve the PDF via the default custom cloud storage disk.
     */
    override fun useStorage(use: Boolean): ConversionParameters {
        return if (use) useStorage { _ -> } else this
    }

    /**
     * Add custom [headers] to the request the PDFire server makes to the provided [url].
     */
    fun withHeaders(headers: Map<String, Any>): ConversionParameters {
        this.headers.putAll(headers)
        return this
    }

    /**
     * Add page ranges that should be captured.
     */
    fun addPageRange(range: String): ConversionParameters {
        pageRanges.add(range)
        return this
    }

    /**
     * Set the [headerTemplate] option.
     */
    fun setHeaderTemplate(html: String): ConversionParameters {
        headerTemplate = html
        return this
    }

    /**
     * Set the [footerTemplate] option.
     */
    fun setFooterTemplate(html: String): ConversionParameters {
        footerTemplate = html
        return this
    }

    /**
     * Set the [waitForSelectorTimeout] option.
     */
    fun setWaitForSelectorTimeout(timeout: Duration): ConversionParameters {
        waitForSelectorTimeout = timeout
        return this
    }

    /**
     * Set the [waitUntilTimeout] option.
     */
    fun setWaitUntilTimeout(timeout: Duration): ConversionParameters {
        waitUntilTimeout = timeout
        return this
    }

    /**
     * Set the [delay] option.
     */
    fun setDelay(delay: Duration): ConversionParameters {
        this.delay = delay
        return this
    }

    /**
     * Set the [timeout] option.
     */
    fun setTimeout(timeout: Duration): ConversionParameters {
        this.timeout = timeout
        return this
    }

    /**
     * Create a map from the parameters.
     */
    override fun toMap(): Map<String, Any?> = mapOf(
        Pair("html", html),
        Pair("url", url),
        Pair("cdn", cdn),
        Pair("landscape", landscape),
        Pair("printBackground", printBackground),
        Pair("scale", scale),
        Pair("format", format?.toString()),
        Pair("paperWidth", paperWidth),
        Pair("paperHeight", paperHeight),
        Pair("margin", margin),
        Pair("marginTop", marginTop),
        Pair("marginRight", marginRight),
        Pair("marginBottom", marginBottom),
        Pair("marginLeft", marginLeft),
        Pair("pageRanges", if (pageRanges.isNotEmpty()) pageRanges.joinToString(",") else null),
        Pair("headerTemplate", headerTemplate),
        Pair("footerTemplate", footerTemplate),
        Pair("preferCSSPageSize", preferCSSPageSize),
        Pair("viewportWidth", viewportWidth),
        Pair("viewportHeight", viewportHeight),
        Pair("blockAds", blockAds),
        Pair("selector", selector),
        Pair("waitForSelector", waitForSelector),
        Pair("waitForSelectorTimeout", waitForSelectorTimeout?.toMillis()),
        Pair("waitUntil", waitUntil.toString()),
        Pair("waitUntilTimeout", waitUntilTimeout?.toMillis()),
        Pair("delay", delay?.toMillis()),
        Pair("timeout", timeout?.toMillis()),
        Pair("headers", headers),
        Pair("emulateMedia", emulateMedia.toString()),
        Pair("ownerPassword", ownerPassword),
        Pair("userPassword", userPassword),
        Pair("allowErrorPage", allowErrorPage),
        Pair("optimize", optimize)
    ).filter { it.value != null }
}
