package io.pdfire.client

/**
 * Configuration for a conversion of multiple documents (merge).
 *
 * Options may require a minimum subscription tier to be usable.
 * See the <a href="https://docs.pdfire.io/options/#cdn">documentation</a> for information on the available options.
 *
 * @property documents Conversion options ([ConversionParameters]) for the documents.
 * @property ownerPassword Secures the PDF with an owner password.
 * @property userPassword Secures the PDF with a user password.
 * @property cdn Indicates if the PDF should be served via the CDN.
 * @property storage Storage configuration for the PDF.
 */
class MergeParameters constructor(
        var documents: MutableList<ConversionParameters> = mutableListOf(),
        var ownerPassword: String? = null,
        var userPassword: String? = null,
        var cdn: Boolean? = null,
        var storage: StorageParameter? = null
): Parameters {
    /**
     * Secure the PDF with an owner / user password.
     */
    fun secure(owner: String? = null, user: String? = null): MergeParameters {
        ownerPassword = if (owner != null) owner else null
        userPassword = if (user != null) user else null
        return this
    }

    /**
     * Set the CDN option.
     */
    override fun setCDN(cdn: Boolean): MergeParameters {
        this.cdn = cdn
        return this
    }

    /**
     * Serve the PDF via the PDFire.io CDN.
     */
    fun useCDN(): MergeParameters {
        return setCDN(true)
    }

    /**
     * Serve the PDF via a custom cloud storage disk.
     */
    override fun useStorage(configure: ((StorageParameter) -> Unit)?): MergeParameters {
        val param = StorageParameter()

        if (configure != null) {
            configure(param)
        }

        storage = param
        return this
    }

    /**
     * Serve the PDF via the default custom cloud storage disk.
     */
    override fun useStorage(use: Boolean): MergeParameters {
        return if (use) useStorage { _ -> } else this
    }

    /**
     * Add documents to the PDF.
     */
    fun addDocuments(vararg documents: ConversionParameters): MergeParameters {
        this.documents.addAll(documents)
        return this
    }

    /**
     * Add a document to the PDF.
     */
    fun addDocument(configure: ((ConversionParameters) -> Unit)? = null): MergeParameters {
        val doc = ConversionParameters()

        if (configure != null) {
            configure(doc)
        }

        documents.add(doc)
        return this
    }

    /**
     * Create a map from the parameters.
     */
    override fun toMap(): Map<String, Any?> = mapOf(
        Pair("documents", documents.map { it.toMap() }),
        Pair("ownerPassword", ownerPassword),
        Pair("userPassword", userPassword),
        Pair("cdn", cdn),
        Pair("storage", storage?.toMap())
    ).filter { it.value != null }
}