package io.pdfire.client

import com.google.gson.Gson
import com.github.kittinunf.fuel.Fuel
import com.github.kittinunf.fuel.core.Response
import com.github.kittinunf.result.failure

/**
 * Sends conversion parameters to the PDFire.io servers and builds the correct result.
 *
 * @property apiKey Your PDFire.io API key.
 */
class Client constructor(var apiKey: String)
{
    companion object {
        /**
         * Base URL to the PDFire.io API.
         */
        @JvmStatic
        val baseUrl = "https://api.pdfire.io/v1"
    }

    private val gson = Gson()

    /**
     * Submit the conversion/merge parameters and return the result.
     */
    fun convert(params: Parameters): Result {
        val body = gson.toJson(params.toMap())

        val (_, response, result) = Fuel
            .post("$baseUrl/conversions")
            .header("Content-Type", "application/json")
            .body(body)
            .response()
        
        when (response.statusCode) {
            400 -> throw BadRequestException(getApiErrors(result.get()))
            401 -> throw UnauthorizedException(getApiErrors(result.get()))
            402 -> throw QuotaException(getApiErrors(result.get()))
            500 -> throw ConversionException(getApiErrors(result.get()))
            201 -> return makeResult(response)
            else -> {
                throw RequestException(getApiErrors(result.get()), "Unknown error.")
            }
        }
    }

    /**
     * Convert the given URL with the provided conversion parameters.
     */
    fun convertURL(url: String, params: ConversionParameters? = null): Result {
        val p = params ?: ConversionParameters()
        p.url = url

        return convert(p)
    }

    /**
     * Convert the given HTML with the provided conversion parameters.
     */
    fun convertHTML(html: String, params: ConversionParameters? = null): Result {
        val p = params ?: ConversionParameters()
        p.html = html

        return convert(p)
    }

    /**
     * Submit the conversion/merge parameters and return the result as a byte array.
     */
    fun convertToByteArray(params: Parameters): ByteArray {
        return convert(params.setCDN(false).useStorage(false)).getBytes()
    }

    /**
     * Submit the conversion/merge parameters and return the conversion result.
     * Throws an error if the account does not include the CDN option.
     */
    fun convertUsingCDN(params: Parameters): Conversion {
        return convert(params.setCDN(true)) as Conversion
    }

    /**
     * Submit the conversion/merge parameters and return the conversion result.
     * Throws a error if the accout does not include the storage option.
     */
    fun convertUsingStorage(params: Parameters, configure: ((StorageParameter) -> Unit)? = null): Conversion {
        return convert(params.useStorage(configure)) as Conversion
    }

    private fun getApiErrors(body: ByteArray): List<ApiError> {
        return gson.fromJson(body.toString(), ErrorResponse::class.java).errors
    }

    private fun makeResult(response: Response): Result {
        val contentTypes = response.headers["Content-Type"].toTypedArray()

        return when (if (contentTypes.isNotEmpty()) contentTypes[0] else "application/octet-stream") {
            "application/json" -> Gson().fromJson(response.body().toString(), Conversion::class.java)
            else -> {
                BytesResult(response.body().toByteArray())
            }
        }
    }
}
