package io.paulbaker.libs.datastructures

import com.fasterxml.jackson.annotation.JsonCreator
import com.fasterxml.jackson.annotation.JsonProperty
import java.util.*

/**
 * Represents a pair of arbitrarily typed items
 *
 * @author Paul Nelson Baker
 * @see <a href="https://github.com/paul-nelson-baker/">GitHub</a>
 * @see <a href="https://www.linkedin.com/in/paul-n-baker/">LinkedIn</a>
 * @since 2019-01
 */
data class Tuple<T, U>(val first: T, val second: U)

/**
 * Represents a pair of arbitrarily (yet comparable) typed items
 *
 * @author Paul Nelson Baker
 * @see <a href="https://github.com/paul-nelson-baker/">GitHub</a>
 * @see <a href="https://www.linkedin.com/in/paul-n-baker/">LinkedIn</a>
 * @since 2019-01
 */
data class OrdinalTuple<T : Comparable<T>, U : Comparable<U>>(val first: T, val second: U) :
    Comparable<OrdinalTuple<T, U>> {
    override fun compareTo(other: OrdinalTuple<T, U>): Int = compareValuesBy(this, other, { it.first }, { it.second })
}

/**
 * Represents a username and password pair
 *
 * @author Paul Nelson Baker
 * @see <a href="https://github.com/paul-nelson-baker/">GitHub</a>
 * @see <a href="https://www.linkedin.com/in/paul-n-baker/">LinkedIn</a>
 * @since 2019-01
 */
data class CredentialsTuple @JsonCreator constructor(
    @JsonProperty("username") val username: String,
    @JsonProperty("password") val password: CharArray
) {
    /**
     * @return A string with a base 64 encoded username and password, prefixed with "Basic "
     */
    fun asBasicAuthString(): String {
        val chars = username.toCharArray() + password
        val bytes = chars.map { it.toByte() }.toByteArray()
        return "Basic ${Base64.getEncoder().encodeToString(bytes)}"
    }

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (javaClass != other?.javaClass) return false

        other as CredentialsTuple

        if (username != other.username) return false
        if (!password.contentEquals(other.password)) return false

        return true
    }

    override fun hashCode(): Int {
        var result = username.hashCode()
        result = 31 * result + password.contentHashCode()
        return result
    }
}