package io.overcoded.grid.processor.column;

import io.overcoded.grid.processor.FieldCollector;
import lombok.RequiredArgsConstructor;

import javax.persistence.ManyToMany;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.OneToOne;
import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Tries to figure out the join column to an entity from another.
 */
@RequiredArgsConstructor
public class JoinFieldFinder {
    private final FieldCollector fieldCollector;
    private final NameTransformer nameTransformer;

    /**
     * Tries to discover join field. It's used in case of context menus.
     * When a field in a parent type is (e.g.) a OneToMany list, that means
     * those items should have a field (generic type) which owns the relationship.
     * This method is trying to find this relationship.
     *
     * @param genericType the type which owns the relationship (in case of a OneToMany list, it's the generic type of the list)
     * @param parentType the type which holds an instance of genericType or a list of them
     * @return Field which is the best candidate
     */
    public Field find(Class<?> genericType, Class<?> parentType) {
        Set<Field> candidates = getCandidates(genericType, parentType);
        return findBest(candidates, parentType);
    }

    private Field findBest(Set<Field> candidates, Class<?> parentType) {
        return hasOneSingleCandidate(candidates)
                ? getAnyCandidate(candidates)
                : findMatching(candidates, parentType)
                .orElseGet(() -> getAnyCandidate(candidates));
    }

    private Optional<Field> findMatching(Set<Field> candidates, Class<?> parentType) {
        String parentField = nameTransformer.decapitalize(parentType.getName());
        return candidates.stream().filter(field -> field.getName().equals(parentField)).findFirst();
    }

    private boolean hasOneSingleCandidate(Set<Field> candidates) {
        return candidates.size() == 1;
    }

    private Field getAnyCandidate(Set<Field> candidates) {
        return candidates.stream().findAny().orElse(null);
    }

    private Set<Field> getCandidates(Class<?> type, Class<?> filterType) {
        return fieldCollector.getFields(type)
                .stream()
                .filter(field -> filterType.equals(getGenericType(field)) || getGenericType(field).isAssignableFrom(filterType))
                .filter(field -> field.isAnnotationPresent(OneToOne.class) || field.isAnnotationPresent(ManyToOne.class) || field.isAnnotationPresent(OneToMany.class) || field.isAnnotationPresent(ManyToMany.class))
                .collect(Collectors.toSet());
    }

    private Class<?> getGenericType(Field field) {
        Class<?> result = field.getType();
        if (field.getType().isAssignableFrom(Collection.class)
                || field.getType().isAssignableFrom(Set.class)
                || field.getType().isAssignableFrom(List.class)) {
            ParameterizedType parameterizedType = (ParameterizedType) field.getGenericType();
            Type[] actualTypeArguments = parameterizedType.getActualTypeArguments();
            result = (Class<?>) actualTypeArguments[0];
        }
        return result;
    }
}
