package io.overcoded.grid.processor;

import org.reflections.Reflections;
import org.reflections.scanners.Scanners;
import org.reflections.util.ClasspathHelper;
import org.reflections.util.ConfigurationBuilder;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * ReflectionsFactory generates Reflections instances.
 * This factory has in-memory cache to store already created
 * Reflections to prevent creating unnecessary instances.
 */
class ReflectionsFactory {
    /**
     * Internal cache for package related reflections
     */
    private final Map<String, Reflections> packageCache = new ConcurrentHashMap<>();
    /**
     * Internal cache for type related reflections
     */
    private final Map<Class<?>, Reflections> typeCache = new ConcurrentHashMap<>();

    /**
     * If not exists, it creates a reflections instance for the
     * specified package.
     * If the reflections instance already created it will be
     * returned.
     *
     * @param basePackage name of the package, which should be scanned by the reflections instance
     * @return the corresponding reflections instance
     */
    Reflections create(String basePackage) {
        return packageCache.computeIfAbsent(basePackage, Reflections::new);
    }

    /**
     * If not exists, it creates a reflections instance for the base
     * package. If the reflections instance already created it will be
     * returned.
     *
     * @return the reflections instance for the root package
     */
    Reflections create() {
        return packageCache.computeIfAbsent("", emptyPackage -> getBaseReflections());
    }

    /**
     * If not exists, it creates a reflections instance for the
     * specified type.
     * If the reflections instance already created it will be
     * returned.
     *
     * @param type which should be scanned by the reflections instance
     * @return the corresponding reflections instance
     */
    Reflections create(Class<?> type) {
        return typeCache.computeIfAbsent(type, this::createReflectionsForType);
    }

    /**
     * Creates the reflections instance for the specified type
     * @param type which should be scanned by the reflections instance
     * @return a new reflections instance
     */
    private Reflections createReflectionsForType(Class<?> type) {
        return new Reflections(new ConfigurationBuilder()
                .setUrls(ClasspathHelper.forClass(type))
                .setScanners(Scanners.FieldsAnnotated, Scanners.TypesAnnotated, Scanners.SubTypes));
    }

    /**
     * Creates the base reflections to the root package.
     * <p>
     * Used for finding GridSystem.
     *
     * @return a new reflections instance
     */
    private Reflections getBaseReflections() {
        return new Reflections(new ConfigurationBuilder()
                .setUrls(ClasspathHelper.forPackage(""))
                .setScanners(Scanners.TypesAnnotated));
    }
}
