package io.overcoded.grid.processor;

import io.overcoded.grid.ColumnInfo;
import io.overcoded.grid.GridInfo;
import io.overcoded.grid.GridMenuGroup;
import io.overcoded.grid.annotation.GridView;
import io.overcoded.grid.processor.column.NameTransformer;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Creates a GridInfo from type which is annotated with GridView.
 */
@Slf4j
@RequiredArgsConstructor
public class GridInfoViewFactory implements GridInfoFactory<GridView> {
    /**
     * Internal cache for GridInfo related to type
     */
    private final Map<Class<?>, GridInfo> typeCache = new ConcurrentHashMap<>();
    private final NameTransformer nameTransformer;
    private final ColumnInfoCollector columnInfoCollector;
    private final GridMenuGroupCollector gridMenuGroupCollector;

    /**
     * Corresponding annotation type. As this implementation based on
     * GridView annotation, this method should return GridView.class.
     *
     * @return expected annotation on input type of create method.
     */
    @Override
    public Class<GridView> getSourceType() {
        return GridView.class;
    }

    /**
     * Creates a GridInfo instance based on the input type GridView annotation.
     * It also sets the columns and groups for the GridInfo.
     *
     * @param type the type which should be transformed into GridInfo
     * @return a GridInfo instance
     * @throws IllegalArgumentException when type is not annotated with GridView
     */
    public GridInfo create(Class<?> type) {
        if (!type.isAnnotationPresent(getSourceType())) {
            throw new IllegalArgumentException("GridView annotation must present on input type.");
        }
        return typeCache.computeIfAbsent(type, this::createGridInfo);
    }

    private GridInfo createGridInfo(Class<?> type) {
        log.debug("Creating grid info for {}", type.getName());
        GridView source = type.getAnnotation(getSourceType());
        List<ColumnInfo> columns = columnInfoCollector.collect(type);
        Map<String, GridMenuGroup> menuGroups = gridMenuGroupCollector.collect(type);
        String name = nameTransformer.transform(type.getSimpleName());
        return GridInfo.builder()
                .columns(columns)
                .formEnabled(!source.hideForm())
                .enabledMethods(List.of(source.enabledMethods()))
                .enabledFor(List.of(source.enabledFor()))
                .description(source.description())
                .sourceAnnotation(getSourceType())
                .menuGroups(menuGroups)
                .type(type)
                .name(name)
                .build();
    }
}
