package io.overcoded.grid;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

import java.util.List;

/**
 * MenuEntry is an entry in the menu group (not grouped) or
 * a submenu under another menu entry (grouped).
 * <p>
 * Grouped menu entry should start with its parent
 * menu entry path.
 */
@Data
@Builder
@NoArgsConstructor
@AllArgsConstructor
public class MenuEntry implements Comparable<MenuEntry> {
    /**
     * Type belongs to this menu entry.
     */
    private Class<?> type;

    /**
     * Icon of the menu entry
     */
    private String icon;

    /**
     * Grouped menu entries are child menu entries.
     * Child menu entries have a parent menu entry.
     */
    private boolean grouped;

    /**
     * Path suffix of the menu entry.
     * This path is the final, full path of the page.
     */
    private String path;

    /**
     * Human-readable form of the menu entry which can be
     * used as title or menu entry label.
     */
    private String label;

    /**
     * Defines the order of the menu entry
     */
    private int order;

    /**
     * Menu entries which belong to this parent menu.
     * This means each MenuEntry in this list should be
     * grouped.
     */
    private List<MenuEntry> entries;

    /**
     * Custom configuration to the menu entry.
     * Originally designed because of dialog grid menu item, but this configuration
     * can be used here too.
     */
    private String configuration;

    /**
     * List of roles who can access this menu entry.
     * <p>
     * Implementation can be different: the menu entry can be disabled
     * or not even visible.
     */
    private List<String> enabledFor;

    @Override
    public int compareTo(MenuEntry other) {
        return Integer.compare(order, other.order);
    }
}
