package io.overcoded.grid.processor;

import io.overcoded.grid.MenuEntry;
import io.overcoded.grid.annotation.GridView;
import io.overcoded.grid.processor.column.NameTransformer;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Creates a MenuEntry from a type, annotated with GridView.
 */
@Slf4j
@RequiredArgsConstructor
public class MenuEntryFactory {
    private final NameTransformer nameTransformer;

    /**
     * Creates a MenuEntry from a type.
     *
     * @param type       which should be processed and annotated with GridView
     * @param pathPrefix prefix of the enclosed MenuGroup.
     * @return MenuEntry created from GridView.
     * @throws IllegalArgumentException in case when GridView annotation is not present.
     */
    public MenuEntry create(Class<?> type, String pathPrefix) {
        if (!type.isAnnotationPresent(GridView.class)) {
            throw new IllegalArgumentException("Input class should be annotated with GridView.");
        }
        log.debug("Creating menu entry for {} under {}", type.getName(), pathPrefix);
        GridView source = type.getAnnotation(GridView.class);
        return MenuEntry
                .builder()
                .type(type)
                .icon(source.icon())
                .order(source.order())
                .entries(new ArrayList<>())
                .grouped(source.grouping())
                .path(pathPrefix + source.path())
                .label(nameTransformer.transformPath(getFinalPathSegment(source)))
                .enabledFor(List.of(source.enabledFor()))
                .build();
    }

    public MenuEntry create(io.overcoded.grid.annotation.MenuEntry entry) {
        MenuEntry result = null;
        if (Objects.nonNull(entry.href())) {
            String url = hasValidUrl(entry) ? entry.href().url() : null;
            result = MenuEntry.builder()
                    .icon(entry.icon())
                    .order(entry.order())
                    .configuration(entry.config())
                    .enabledFor(List.of(entry.enabledFor()))
                    .label(entry.label())
                    .entries(List.of())
                    .path(url)
                    .build();
        }
        return result;
    }

    private boolean hasValidUrl(io.overcoded.grid.annotation.MenuEntry entry) {
        return Objects.nonNull(entry.href().url()) && !entry.href().url().isBlank();
    }

    private String getFinalPathSegment(GridView source) {
        return source.path().substring(source.path().lastIndexOf("/") + 1);
    }
}
