package io.overcoded.grid.processor;

import io.overcoded.grid.ColumnInfo;
import io.overcoded.grid.annotation.FieldProvider;
import io.overcoded.grid.annotation.GridColumn;
import io.overcoded.grid.annotation.ImagePreview;
import io.overcoded.grid.annotation.ReadOnly;
import io.overcoded.grid.processor.column.DisplayValueExpressionExtractor;
import io.overcoded.grid.processor.column.FieldProviderTypeDecider;
import io.overcoded.grid.processor.column.NameTransformer;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.Field;
import java.util.Optional;

import static java.util.Arrays.asList;

/**
 * Creates a ColumnInfo instance from Field
 */
@Slf4j
@RequiredArgsConstructor
public class ColumnInfoFactory {
    private final FieldProviderTypeDecider fieldProviderTypeDecider;
    private final DisplayValueExpressionExtractor displayValueExpressionExtractor;
    private final GridDialogValidator gridDialogValidator;
    private final NameTransformer nameTransformer;

    /**
     * Creates ColumnInfo from Field. Field should be member of parentType.
     * <p>
     * `@OneToMany` and `@ManyToMany` fields are not considered as fields,
     * because they will appear in dialogs.
     * <p>
     * Fields with transient keyword or annotated with @Transient can
     * visible in the grid, but shouldn't appear on the form.
     *
     * @param field      which should be converted into ColumnInfo
     * @param parentType which owns the field
     * @return ColumnInfo instance if the conditions are met, null otherwise.
     */
    public ColumnInfo create(Field field, Class<?> parentType) {
        log.debug("Creating column info for {} in {}", field.getName(), parentType.getName());
        ColumnInfo columnInfo = null;
        if (!gridDialogValidator.isDialogCandidate(field)) {
            Optional<GridColumn> gridColumn = Optional.ofNullable(field.getAnnotation(GridColumn.class));
            Optional<ImagePreview> imagePreview = Optional.ofNullable(field.getAnnotation(ImagePreview.class));
            columnInfo = ColumnInfo
                    .builder()
                    .type(field.getType())
                    .name(field.getName())
                    .parentType(parentType)
                    .label(nameTransformer.transform(field))
                    .readOnly(field.isAnnotationPresent(ReadOnly.class))
                    .fieldProviderType(fieldProviderTypeDecider.decide(field, gridColumn.map(GridColumn::fieldProvider).orElse(null)))
                    .imagePreview(imagePreview.map(this::toImagePreview).orElse(null))
                    .fieldProviderConfiguration(gridColumn.map(GridColumn::fieldProvider).map(FieldProvider::configuration).orElse("DEFAULT"))
                    .hidden(gridColumn.map(GridColumn::hidden).orElse(true))
                    .filter(gridColumn.map(GridColumn::filter).orElse(false))
                    .mainFilter(gridColumn.map(GridColumn::mainFilter).orElse(false))
                    .order(gridColumn.map(GridColumn::order).orElse(Integer.MAX_VALUE))
                    .placeholder(gridColumn.map(GridColumn::placeholder).orElse(null))
                    .displayValueExpressionParts(displayValueExpressionExtractor.extract(field))
                    .build();
        }
        return columnInfo;
    }

    private io.overcoded.grid.ImagePreview toImagePreview(ImagePreview imagePreview) {
        return io.overcoded.grid.ImagePreview.builder()
                .link(imagePreview.url())
                .arguments(asList(imagePreview.args()))
                .build();
    }
}
