package io.overcoded.grid.processor.column;

import io.overcoded.grid.annotation.FieldProviderType;
import lombok.RequiredArgsConstructor;
import org.springframework.data.repository.support.Repositories;

import java.lang.reflect.Field;
import java.util.Objects;

/**
 * FieldProviderTypeEvaluator for JPA_REPOSITORY instance of FieldProviderType.
 * Only types which have corresponding JpaRepository.
 */
@RequiredArgsConstructor
public class JpaRepositoryFieldProviderTypeEvaluator implements FieldProviderTypeEvaluator {
    private final Repositories repositories;

    /**
     * The FieldProviderType instance bound to this evaluator.
     *
     * @return JPA_REPOSITORY
     */
    @Override
    public FieldProviderType getType() {
        return FieldProviderType.JPA_REPOSITORY;
    }

    /**
     * Tests that a field can be marked with JPA_REPOSITORY FieldProviderType.
     * <p>
     * This implementation uses Spring's Repositories class, which can provide
     * the information that a type has a repository or not.
     *
     * @param field which should be examined
     * @return true if the field which has managed JpaRepository interface.
     */
    @Override
    public boolean evaluate(Field field) {
        return Objects.nonNull(field)
                && repositories.hasRepositoryFor(field.getType());
    }

    /**
     * As this field provider type evaluator checks only for types which has
     * JpaRepository interfaces, we should evaluate this only after
     * PickerFieldProviderTypeEvaluator.
     *
     * @return 100
     */
    @Override
    public int getOrder() {
        return 100;
    }
}
