package io.overcoded.grid.processor;

import io.overcoded.grid.GridMenuGroup;
import io.overcoded.grid.GridMenuItem;
import io.overcoded.grid.annotation.MenuGroup;
import io.overcoded.grid.annotation.MenuGroups;
import lombok.RequiredArgsConstructor;

import java.util.*;

import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toMap;

@RequiredArgsConstructor
public class GridMenuGroupCollector {
    private final GridMenuGroupFactory groupFactory;
    private final GridMenuEntryCollector gridMenuEntryCollector;

    public Map<String, GridMenuGroup> collect(Class<?> type) {
        Map<String, GridMenuGroup> emptyGroups = getEmptyGroups(type);
        return getMenuGroups(emptyGroups, type);
    }

    private Map<String, GridMenuGroup> getMenuGroups(Map<String, GridMenuGroup> menuGroups, Class<?> type) {
        List<GridMenuItem> menuItems = gridMenuEntryCollector.collect(type);
        for (GridMenuItem menuItem : menuItems) {
            String key = Objects.nonNull(menuItem.getGroup()) ? menuItem.getGroup() : "";
            GridMenuGroup group = menuGroups.containsKey(key) ? menuGroups.get(key) : getAliasGroup(menuGroups, key);
            if (Objects.isNull(group.getItems())) {
                group.setItems(new ArrayList<>());
            }
            group.getItems().add(menuItem);
        }
        menuGroups.values().stream().map(GridMenuGroup::getItems).forEach(Collections::sort);
        return menuGroups;
    }

    private GridMenuGroup getAliasGroup(Map<String, GridMenuGroup> menuGroups, String key) {
        return menuGroups.values().stream()
                .filter(group -> group.getAliasFor().contains(key))
                .findFirst()
                .orElseGet(() -> menuGroups.get(""));
    }

    private Map<String, GridMenuGroup> getEmptyGroups(Class<?> type) {
        Map<String, GridMenuGroup> groups = new TreeMap<>();
        if (type.isAnnotationPresent(MenuGroup.class)) {
            groups = extract(type.getAnnotationsByType(MenuGroup.class));
        } else if (type.isAnnotationPresent(MenuGroups.class)) {
            groups = extract(type.getAnnotation(MenuGroups.class).value());
        }
        if (!groups.containsKey("")) {
            groups.put("", GridMenuGroup.builder().order(Integer.MIN_VALUE).items(new ArrayList<>()).aliasFor(Set.of()).build());
        }
        return groups;
    }

    private Map<String, GridMenuGroup> extract(MenuGroup[] menuGroups) {
        return Arrays.stream(menuGroups)
                .map(groupFactory::create)
                .sorted()
                .collect(toMap(GridMenuGroup::getName, identity(), (first, second) -> first, TreeMap::new));
    }

}
