package io.overcoded.grid;

import io.overcoded.grid.annotation.FieldProviderType;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

import java.util.List;

/**
 * Column info collects all the necessary information to
 * configure a grid and the corresponding form.
 */
@Data
@Builder
@NoArgsConstructor
@AllArgsConstructor
public class ColumnInfo implements Comparable<ColumnInfo> {
    /**
     * Defines that the column should be visible or hidden in the grid.
     */
    private boolean hidden;

    /**
     * Marks the column as a filter, which means the grid should
     * filter the values based on this field.
     */
    private boolean filter;

    /**
     * Main filter is used to define the primary filter used in cases
     * when something else needed to be filtered by this type and we
     * want to filter its ui element (e.g. combobox).
     */
    private boolean mainFilter;

    /**
     * Title of the column and the label of the input field
     */
    private String label;

    /**
     * Name of the column, normally it's the name of the field
     * in the parent class.
     */
    private String name;

    /**
     * Type of the field.
     */
    private Class<?> type;

    /**
     * Type of the class which contains this field
     */
    private Class<?> parentType;

    /**
     * Placeholder / helper text when the field is empty.
     */
    private String placeholder;

    /**
     * Order of the column in the grid and in the form.
     */
    private int order;

    /**
     * Field provider type defines what kind of control element
     * should we display for the view.
     */
    private FieldProviderType fieldProviderType;

    /**
     * If ImagePreview annotation present on the column, we are
     * populating this field.
     * <p>
     * Implementation can be different, our idea is to have popover
     * on the field if this field exists.
     */
    private ImagePreview imagePreview;

    /**
     * Some field providers need to be configured to the field
     * E.g.: source code editor or rich text editor.
     * This field can be used to pass the configuration.
     * <p>
     * Designed to use as an enum, but keeping it String to give
     * the flexibility to the client.
     */
    private String fieldProviderConfiguration;

    /**
     * Display value expression parts are coming from
     * DisplayValue annotation.
     * <p>
     * Originally designed to use with SpEL, but any other
     * expression language can be used. In the implementation
     * only the following rules should be considered:
     * <p>
     * If an element starts with `'` that means the element is
     * a string and should be concatenated to the other elements
     * without any modification.
     * <p>
     * If an element starts with anything else it should be considered
     * as variable which should available in the context. As the context
     * is not containing the variable, the parent type should be used
     * as prefix.
     */
    private List<String> displayValueExpressionParts;

    /**
     * Natural order of ColumnInfo
     * @param other the object to be compared.
     * @return natural order based on order field
     */
    @Override
    public int compareTo(ColumnInfo other) {
        return Integer.compare(order, other.order);
    }
}
