package io.overcoded.grid.processor.column;

import java.lang.reflect.Field;
import java.util.Objects;

/**
 * Should humanize field names.
 */
public class NameTransformer {
    /**
     * Transforms a field into human-readable form.
     * The implementation based on <a href="https://stackoverflow.com/a/2560017">this answer</a>.
     *
     * @param field which name should be transformed
     * @return the human-readable form of the name of the field.
     */
    public String transform(Field field) {
        return transform(field.getName());
    }

    /**
     * Transforms a url into human-readable form.
     * The implementation based on <a href="https://stackoverflow.com/a/2560017">this answer</a>.
     *
     * @param text which should be transformed
     * @return the human-readable form of the type
     */
    public String transformPath(String text) {
        return transform(text.replaceAll("[-_/]", " "));
    }

    /**
     * Transforms a text into human-readable form.
     * The implementation based on <a href="https://stackoverflow.com/a/2560017">this answer</a>.
     *
     * @param text which should be transformed
     * @return the human-readable form of the type
     */
    public String transform(String text) {
        return splitCamelCase(capitalize(text));
    }

    private String splitCamelCase(String input) {
        return input.replaceAll(
                String.format("%s|%s|%s",
                        "(?<=[A-Z])(?=[A-Z][a-z])",
                        "(?<=[^A-Z])(?=[A-Z])",
                        "(?<=[A-Za-z])(?=[^A-Za-z])"
                ),
                " "
        );
    }

    /**
     * Capitalize makes the first character uppercase.
     * @param input string which should be capitalized
     * @return the input string with uppercase first letter
     */
    public String capitalize(String input) {
        String result = input;
        if (Objects.nonNull(input) && input.length() >= 1) {
            result = input.substring(0, 1).toUpperCase() + input.substring(1);
        }
        return result;
    }

    /**
     * Decapitalize makes the first character lowercase.
     * @param input string which should be decapitalized
     * @return the input string with lowercase first letter
     */
    public String decapitalize(String input) {
        String result = input;
        if (Objects.nonNull(input) && input.length() >= 1) {
            result = input.substring(0, 1).toUpperCase() + input.substring(1);
        }
        return result;
    }
}
