package io.overcoded.grid.processor;

import io.overcoded.grid.CustomViewGridMenuItem;
import io.overcoded.grid.DialogGridMenuItem;
import io.overcoded.grid.GridMenuItem;
import io.overcoded.grid.LinkGridMenuItem;
import io.overcoded.grid.annotation.GridDialog;
import io.overcoded.grid.annotation.MenuEntry;
import io.overcoded.grid.annotation.WithCustomFilter;
import io.overcoded.grid.processor.column.JoinFieldFinder;
import io.overcoded.grid.processor.column.NameTransformer;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.*;

@Slf4j
@RequiredArgsConstructor
public class GridMenuEntryFactory {
    private final JoinFieldFinder joinFieldFinder;
    private final NameTransformer nameTransformer;

    /**
     * Creates a custom view grid menu item or a link grid menu item
     *
     * @param menuEntry
     * @return a menu item or null if href is not present
     */
    public GridMenuItem create(MenuEntry menuEntry) {
        GridMenuItem result = null;
        if (!isDialogGridMenuItem(menuEntry)) {
            if (isLinkGridMenuItem(menuEntry)) {
                result = createLinkGridMenuItem(menuEntry);
            } else {
                result = createCustomViewGridMenuItem(menuEntry);
            }
        }
        return result;
    }

    private boolean isDialogGridMenuItem(MenuEntry menuEntry) {
        return Objects.isNull(menuEntry.href()) || (!isCustomViewGridMenuItem(menuEntry) && !isLinkGridMenuItem(menuEntry));
    }

    private boolean isCustomViewGridMenuItem(MenuEntry menuEntry) {
        return Objects.nonNull(menuEntry.href().view()) && !Void.class.equals(menuEntry.href().view());
    }

    private boolean isLinkGridMenuItem(MenuEntry menuEntry) {
        return Objects.nonNull(menuEntry.href().url()) && !menuEntry.href().url().isBlank();
    }

    public GridMenuItem create(Field field, Class<?> parentType) {
        return create(field, getGenericType(field), parentType);
    }

    public GridMenuItem create(Field field, Class<?> genericType, Class<?> parentType) {
        DialogGridMenuItem result = null;
        if (genericType.isAnnotationPresent(GridDialog.class)) {
            GridDialog dialog = genericType.getAnnotation(GridDialog.class);
            result = createDialogGridMenuItem(field, genericType, parentType)
                    .description(dialog.description())
                    .build();
            updateEnabledFor(result, dialog);
        }
        return result;
    }

    private CustomViewGridMenuItem createCustomViewGridMenuItem(MenuEntry entry) {
        return CustomViewGridMenuItem.builder()
                .icon(entry.icon())
                .order(entry.order())
                .label(entry.label())
                .divided(entry.divided())
                .view(entry.href().view())
                .target(entry.href().target())
                .group(entry.menuGroup())
                .arguments(List.of(entry.href().args()))
                .enabledFor(List.of(entry.enabledFor()))
                .build();
    }

    private GridMenuItem createLinkGridMenuItem(MenuEntry entry) {
        return LinkGridMenuItem.builder()
                .icon(entry.icon())
                .order(entry.order())
                .label(entry.label())
                .divided(entry.divided())
                .link(entry.href().url())
                .target(entry.href().target())
                .group(entry.menuGroup())
                .arguments(List.of(entry.href().args()))
                .enabledFor(List.of(entry.enabledFor()))
                .build();
    }

    private DialogGridMenuItem.DialogGridMenuItemBuilder<?, ?> createDialogGridMenuItem(Field field, Class<?> genericType, Class<?> parentType) {
        DialogGridMenuItem.DialogGridMenuItemBuilder<?, ?> builder = DialogGridMenuItem.builder();
        if (genericType.isAnnotationPresent(MenuEntry.class)) {
            builder = withEntry(field, genericType.getAnnotation(MenuEntry.class));
        } else {
            builder = withoutEntry(field, genericType.getAnnotation(GridDialog.class));
        }
        return builder
                .type(genericType)
                .customFilter(hasCustomFilter(field))
                .name(getName(genericType, parentType))
                .parentFieldName(getParentFieldName(field));
    }

    private DialogGridMenuItem.DialogGridMenuItemBuilder<?, ?> withEntry(Field field, MenuEntry entry) {
        return DialogGridMenuItem.builder()
                .icon(entry.icon())
                .order(entry.order())
                .divided(entry.divided())
                .label(getLabel(field, entry))
                .group(entry.menuGroup())
                .enabledFor(List.of(entry.enabledFor()));
    }

    private DialogGridMenuItem.DialogGridMenuItemBuilder<?, ?> withoutEntry(Field field, GridDialog dialog) {
        return DialogGridMenuItem.builder()
                .description(dialog.description())
                .label(getLabel(field, null))
                .enabledFor(List.of(dialog.enabledFor()));
    }

    private String getLabel(Field field, MenuEntry entry) {
        return hasLabel(entry) ? entry.label() : nameTransformer.transform(field);
    }

    private boolean hasLabel(MenuEntry entry) {
        return Objects.nonNull(entry) && Objects.nonNull(entry.label()) && !entry.label().isBlank();
    }

    private String getName(Class<?> genericType, Class<?> parentType) {
        Field joinField = joinFieldFinder.find(genericType, parentType);
        return Objects.nonNull(joinField) ? joinField.getName() : null;
    }

    private String getParentFieldName(Field field) {
        String parentFieldName = null;
        if (hasCustomFilter(field)) {
            parentFieldName = field.getAnnotation(WithCustomFilter.class).value();
        }
        return parentFieldName;
    }

    private boolean hasCustomFilter(Field field) {
        return field.isAnnotationPresent(WithCustomFilter.class);
    }

    private void updateEnabledFor(DialogGridMenuItem result, GridDialog dialog) {
        Set<String> enabledFor = Objects.nonNull(result.getEnabledFor()) ? new HashSet<>(result.getEnabledFor()) : new HashSet<>();
        enabledFor.addAll(List.of(dialog.enabledFor()));
        result.setEnabledFor(new ArrayList<>(enabledFor));
    }

    private Class<?> getGenericType(Field field) {
        Class<?> result = field.getType();
        if (field.getType().isAssignableFrom(Collection.class)
                || field.getType().isAssignableFrom(Set.class)
                || field.getType().isAssignableFrom(List.class)) {
            ParameterizedType parameterizedType = (ParameterizedType) field.getGenericType();
            Type[] actualTypeArguments = parameterizedType.getActualTypeArguments();
            result = (Class<?>) actualTypeArguments[0];
        }
        return result;
    }
}
