package io.overcoded.grid.processor;

import io.overcoded.grid.GridInfo;
import lombok.RequiredArgsConstructor;

import java.lang.annotation.Annotation;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

/**
 * Creates GridInfo from a type.
 * @see GridInfoViewFactory
 * @see GridInfoDialogFactory
 */
@RequiredArgsConstructor
public class DynamicGridInfoFactory {
    private final List<GridInfoFactory<?>> factories;

    /**
     * Creates a GridInfo based on the type.
     * As dialog and view can contain grids, the result depends on which annotation
     * is used for the type.
     * So if the type is annotated with GridView, GridInfoViewFactory will be called.
     * If it's annotated with GridDialog, GridInfoDialogFactory will be called.
     * If a type has both annotation, first found will be called, order is not guaranteed.
     *
     * @param type which should be converted into GridInfo
     * @return GridInfo instance
     */
    public GridInfo create(Class<?> type) {
        return Arrays.stream(type.getAnnotations())
                .map(this::getFactoryForAnnotation)
                .filter(Objects::nonNull)
                .findFirst()
                .map(factory -> factory.create(type))
                .orElse(null);
    }

    private GridInfoFactory<?> getFactoryForAnnotation(Annotation annotation) {
        return factories.stream()
                .filter(factory -> Objects.equals(factory.getSourceType(), annotation.annotationType()))
                .findFirst()
                .orElse(null);
    }
}
