package io.overcoded.grid.processor;

import io.overcoded.grid.ContextMenuGroup;
import io.overcoded.grid.annotation.GridDialog;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Set;

/**
 * Creates a ContextMenuGroup from a Field
 */
@Slf4j
@RequiredArgsConstructor
public class ContextMenuGroupFactory {
    private final ContextMenuEntryFactory contextMenuEntryFactory;
    private final GridDialogValidator gridDialogValidator;

    /**
     * Converts a Field into ContextMenuGroup.
     * The outcome will be a ContextMenuGroup with one single element.
     * Groups with the same name should be merged by ContextMenuGroupCollector.
     * <p>
     *
     * @param field      which should be converted into ContextMenuGroup
     * @param parentType type which contains the Field
     * @return a ContextMenuGroup if the field can be a dialog, null otherwise.
     * @see GridDialogValidator
     */
    public ContextMenuGroup create(Field field, Class<?> parentType) {
        ContextMenuGroup columnInfo = null;
        if (Objects.nonNull(field) && gridDialogValidator.isDialogCandidate(field)) {
            log.debug("Creating dialog for {} in {}", field.getName(), parentType.getName());
            Class<?> genericType = getGenericType(field);
            if (genericType.isAnnotationPresent(GridDialog.class)) {
                GridDialog gridDialog = genericType.getAnnotation(GridDialog.class);
                columnInfo = ContextMenuGroup
                        .builder()
                        .order(gridDialog.order())
                        .icon(gridDialog.groupIcon())
                        .label(gridDialog.menuGroup())
                        .contextMenuEntries(List.of(contextMenuEntryFactory.create(field, genericType, parentType)))
                        .build();
            } else {
                log.warn("{} in {} is a valid candidate as a dialog, but GridDialog annotation is not present on its type.", field.getName(), parentType.getName());
            }
        } else {
            log.debug("Field {} in {} is not a valid dialog candidate", field.getName(), parentType.getName());
        }
        return columnInfo;
    }

    private Class<?> getGenericType(Field field) {
        Class<?> result = field.getType();
        if (field.getType().isAssignableFrom(Collection.class)
                || field.getType().isAssignableFrom(Set.class)
                || field.getType().isAssignableFrom(List.class)) {
            ParameterizedType parameterizedType = (ParameterizedType) field.getGenericType();
            Type[] actualTypeArguments = parameterizedType.getActualTypeArguments();
            result = (Class<?>) actualTypeArguments[0];
        }
        return result;
    }
}
