package io.overcoded.grid.processor;

import io.overcoded.grid.ContextMenuEntry;
import io.overcoded.grid.ContextMenuGroup;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * Collects the ContextMenuGroups for a type.
 */
@Slf4j
@RequiredArgsConstructor
public class ContextMenuGroupCollector {

    private final FieldCollector fieldCollector;
    private final ContextMenuGroupFactory contextMenuGroupFactory;

    /**
     * Collects all the available ContextMenuGroups for the type.
     * First it collects all the fields, converting each of them
     * to ContextMenuGroup, and it is merging them.
     * <p>
     * For the first time, each ContextMenuGroup has one single
     * element.
     * Merge means if the type has multiple ContextMenuGroups with
     * the same name, it will merge its entries.
     *
     * @param type which should be checked
     * @return List of ContextMenuGroups found for the type.
     */
    public List<ContextMenuGroup> collect(Class<?> type) {
        log.debug("Collecting context menu groups for {}", type.getName());
        List<ContextMenuGroup> result = fieldCollector.getFields(type)
                .stream()
                .map(field -> contextMenuGroupFactory.create(field, type))
                .filter(Objects::nonNull)
                .collect(Collectors.toMap(ContextMenuGroup::getLabel, Function.identity(), this::merge, HashMap::new))
                .values()
                .stream()
                .sorted()
                .collect(Collectors.toList());
        log.trace("Collected context menu groups: {}", result);
        return result;
    }

    private ContextMenuGroup merge(ContextMenuGroup stored, ContextMenuGroup fresh) {
        List<ContextMenuEntry> storedMenuEntries = new ArrayList<>(stored.getContextMenuEntries());
        storedMenuEntries.addAll(fresh.getContextMenuEntries());
        storedMenuEntries.sort(null);
        stored.setContextMenuEntries(storedMenuEntries);
        return stored;
    }
}
