package io.overcoded.grid.processor;

import io.overcoded.grid.ColumnInfo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

/**
 * Collect a type's field and converts into ColumnInfo
 */
@Slf4j
@RequiredArgsConstructor
public class ColumnInfoCollector {
    /**
     * Internal cache for columns related to type
     */
    private final Map<Class<?>, List<ColumnInfo>> typeCache = new ConcurrentHashMap<>();
    private final ColumnInfoFactory columnInfoFactory;
    private final FieldCollector fieldCollector;

    /**
     * Collect all fields of a type and converts into ColumnInfo.
     *
     * @param type a type which column should be collected
     * @return List of column info of the type
     */
    public List<ColumnInfo> collect(Class<?> type) {
        return typeCache.computeIfAbsent(type, this::collectColumnInfos);
    }

    private List<ColumnInfo> collectColumnInfos(Class<?> type) {
        log.debug("Collecting column infos for {}", type.getName());
        return fieldCollector.getFields(type)
                .stream()
                .map(field -> columnInfoFactory.create(field, type))
                .filter(Objects::nonNull)
                .sorted()
                .collect(Collectors.toList());
    }
}
