package io.overcoded.grid.processor.dialog;

import io.overcoded.grid.DialogType;
import io.overcoded.grid.processor.FieldCollector;
import io.overcoded.grid.processor.GridDialogValidator;
import lombok.RequiredArgsConstructor;

import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * Decides which DialogType should be applied to a field.
 */
@RequiredArgsConstructor
public class DialogTypeDecider {
    private final FieldCollector fieldCollector;
    private final GridDialogValidator gridDialogValidator;
    private final List<DialogTypeEvaluator> dialogTypeEvaluators;

    /**
     * Made a decision about which dialog type should be applied for the specified type.
     *
     * @param type       which should be evaluated
     * @param parentType which owns the type
     * @return DialogType of the type
     * @see GridDialogValidator
     * @see DialogTypeEvaluator
     */
    public DialogType decide(Class<?> type, Class<?> parentType) {
        return fieldCollector.getFields(parentType)
                .stream()
                .filter(field -> getMatchingField(field, type))
                .filter(gridDialogValidator::isDialogCandidate)
                .map(this::toDialogType)
                .filter(Optional::isPresent)
                .map(Optional::get)
                .findFirst()
                .orElse(DialogType.UNKNOWN);
    }

    private Optional<DialogType> toDialogType(Field field) {
        return dialogTypeEvaluators.stream()
                .filter(evaluator -> evaluator.evaluate(field))
                .findFirst()
                .map(DialogTypeEvaluator::getType);
    }

    private boolean getMatchingField(Field field, Class<?> expectedType) {
        boolean result = field.getType().equals(expectedType);
        if (field.getType().isAssignableFrom(Collection.class)
                || field.getType().isAssignableFrom(Set.class)
                || field.getType().isAssignableFrom(List.class)) {
            Class<?> actualType = getGenericType(field);
            result = expectedType.equals(actualType);
        }
        return result;
    }

    private Class<?> getGenericType(Field field) {
        ParameterizedType parameterizedType = (ParameterizedType) field.getGenericType();
        Type[] actualTypeArguments = parameterizedType.getActualTypeArguments();
        return (Class<?>) actualTypeArguments[0];
    }
}
