package io.overcoded.grid.processor.column;

import io.overcoded.grid.annotation.FieldProviderType;
import io.overcoded.grid.annotation.GridColumn;
import lombok.RequiredArgsConstructor;

import java.lang.reflect.Field;
import java.util.List;
import java.util.Objects;

import static io.overcoded.grid.annotation.FieldProviderType.DEFAULT;

/**
 * Decides which FieldProviderType should be applied on a field.
 */
@RequiredArgsConstructor
public class FieldProviderTypeDecider {
    private final List<FieldProviderTypeEvaluator> fieldProviderTypeEvaluators;

    /**
     * Made a decision about which field provider type should be applied for the specified field.
     * If GridColumn annotation is not present on a field, or it's defined as DEFAULT, we are going
     * through the ordered list of FieldProviderTypeEvaluators, and selecting the first which
     * applied. If there is no evaluator, which resolve the current field, the value remain DEFAULT,
     * which means we let the framework decide.
     * <p>
     * If the GridColumn annotation is present, and it has any other value than AUTO, we are going
     * to use that, because that's the user specific request.
     *
     * @param field      which field provider type should be evaluated
     * @param gridColumn a possible null annotation on the field
     * @return FieldProviderType which should apply for the field.
     */
    public FieldProviderType decide(Field field, GridColumn gridColumn) {
        return Objects.isNull(gridColumn) || gridColumn.fieldProviderType() == DEFAULT
                ? calculateFieldProvider(field)
                : gridColumn.fieldProviderType();
    }

    private FieldProviderType calculateFieldProvider(Field field) {
        return fieldProviderTypeEvaluators.stream()
                .filter(evaluator -> evaluator.evaluate(field))
                .map(FieldProviderTypeEvaluator::getType)
                .findFirst()
                .orElse(DEFAULT);
    }
}
