package io.overcoded.grid.processor;

import io.overcoded.grid.ColumnInfo;
import io.overcoded.grid.MethodInfo;
import io.overcoded.grid.MethodInfoProperties;
import io.overcoded.grid.annotation.GridDialog;
import io.overcoded.grid.annotation.GridView;
import io.overcoded.grid.processor.column.NameTransformer;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.repository.support.Repositories;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

import static java.util.stream.Collectors.toList;


/**
 * Creates MethodInfo for type annotated with GridDialog or GridView.
 * <p>
 * This factory tries to find the best matching method based on filter columns.
 */
@Slf4j
@RequiredArgsConstructor
public class MethodInfoFactory {
    private final Repositories repositories;
    private final NameTransformer nameTransformer;
    private final MethodInfoProperties methodInfoProperties;

    /**
     * Creates a MethodInfo instance based on the input type (should be annotated with GridColumn or GridDialog)
     * and the columns related to the grid.
     *
     * @param type    the type which should be transformed into GridInfo
     * @param columns collected column infos for the type
     * @return a MethodInfo instance or null if no repository found for type
     * @throws IllegalArgumentException when type is not annotated with GridDialog or GridView
     */
    public MethodInfo create(Class<?> type, List<ColumnInfo> columns) {
        if (!type.isAnnotationPresent(GridDialog.class) && !type.isAnnotationPresent(GridView.class)) {
            throw new IllegalArgumentException("GridDialog or GridView annotation must present on input type.");
        }
        Optional<Object> repositoryBean = repositories.getRepositoryFor(type);
        MethodInfo result = null;
        if (repositoryBean.isPresent()) {
            JpaRepository<?, ?> repository = (JpaRepository<?, ?>) repositoryBean.get();
            List<ColumnInfo> filterColumns = columns.stream().filter(ColumnInfo::isFilter).collect(toList());
            String countFallback = methodInfoProperties.getFallbackCountMethod();
            String findFallback = methodInfoProperties.getFallbackFindMethod();
            List<Class<?>> argumentTypes = getArgumentTypes(filterColumns);
            List<String> fieldNames = getFieldNames(filterColumns);
            if (columns.isEmpty()) {
                result = createMethodInfo(repository, findFallback, countFallback, List.of());
            } else {
                List<String> methods = getMethodCandidates(type, argumentTypes, fieldNames);
                String findMethodName = getFindMethodName(methods, findFallback);
                String countMethodName = getCountMethodName(methods, findMethodName, countFallback);
                result = createMethodInfo(repository, findMethodName, countMethodName, argumentTypes);
            }
        }
        return result;
    }

    private String getCountMethodName(List<String> methods, String findMethodName, String countFallback) {
        String countMethodCandidate = findMethodName.replace(methodInfoProperties.getFindMethodPrefix(), methodInfoProperties.getCountMethodPrefix());
        String countMethodName = methods.stream().filter(method -> method.equals(countMethodCandidate)).findFirst().orElse(countFallback);
        if (!countMethodCandidate.equals(countMethodName)) {
            log.warn("Matching count method not found for {}. You may missed to add @FindAllBy or @FindAll annotations.", findMethodName);
        }
        return countMethodName;
    }

    private String getFindMethodName(List<String> methods, String findFallback) {
        return methods.stream()
                .filter(method -> method.startsWith(methodInfoProperties.getFindMethodPrefix()))
                .findFirst()
                .orElse(findFallback);
    }

    private List<String> getMethodCandidates(Class<?> entityType, List<Class<?>> argumentTypes, List<String> fieldNames) {
        return repositories.getRepositoryFor(entityType)
                .stream()
                .map(Object::getClass)
                .flatMap(this::getMethodStream)
                .filter(this::isValidFindAllOrCountAllCandidate)
                .filter(method -> containsArgumentsInExactOrder(List.of(method.getParameterTypes()), argumentTypes))
                .map(Method::getName)
                .filter(name -> containsFieldNamesInExactOrder(name, fieldNames))
                .collect(toList());
    }

    private Stream<Method> getMethodStream(Class<?> repositoryType) {
        return Stream.of(repositoryType.getDeclaredMethods());
    }

    private boolean isValidFindAllOrCountAllCandidate(Method method) {
        return isCountMethod(method) || isFindMethod(method);
    }

    /**
     * Count methods should start with count method prefix
     *
     * @param method which should be checked
     * @return true is method name starts with the predefined count method prefix
     * @see MethodInfoProperties
     */
    private boolean isCountMethod(Method method) {
        return method.getName().startsWith(methodInfoProperties.getCountMethodPrefix());
    }

    /**
     * Find methods should start with find method prefix and should be pageable.
     *
     * @param method which should be checked
     * @return true is method name starts with the predefined find method prefix
     * @see MethodInfoProperties
     */
    private boolean isFindMethod(Method method) {
        return method.getName().startsWith(methodInfoProperties.getFindMethodPrefix())
                && isPageable(method);
    }

    /**
     * A method is pageable when the last parameter of the method is Pageable.
     *
     * @param method which should be checked
     * @return true when the last parameter is Pageable, false otherwise.
     */
    private boolean isPageable(Method method) {
        List<Class<?>> parameterTypes = List.of(method.getParameterTypes());
        return parameterTypes.get(parameterTypes.size() - 1).equals(Pageable.class);
    }

    private List<String> getFieldNames(List<ColumnInfo> filterColumns) {
        return filterColumns.stream().map(ColumnInfo::getName).collect(toList());
    }

    private List<Class<?>> getArgumentTypes(List<ColumnInfo> filterColumns) {
        List<Class<?>> argumentTypes = filterColumns.stream().map(ColumnInfo::getType).collect(java.util.stream.Collectors.toCollection(ArrayList::new));
        argumentTypes.add(Pageable.class);
        return argumentTypes;
    }

    private boolean containsArgumentsInExactOrder(List<Class<?>> methodArgumentTypes, List<Class<?>> expectedArgumentTypes) {
        boolean result = false;
        int expectedSize = expectedArgumentTypes.size();
        int argumentSize = methodArgumentTypes.size();
        if (expectedSize == argumentSize || expectedSize == argumentSize + 1) {
            int i = 0;
            while (i < argumentSize && expectedArgumentTypes.get(i).equals(methodArgumentTypes.get(i))) {
                i++;
            }
            result = !(i < argumentSize);
        }
        return result;
    }

    private boolean containsFieldNamesInExactOrder(String name, List<String> fieldNames) {
        boolean matching = true;
        String separator = methodInfoProperties.getCommonPrefixSeparator();
        List<String> argumentNames = fieldNames.stream().map(nameTransformer::capitalize).collect(toList());
        // remove prefixes
        String simplifiedName = name.substring(name.indexOf(separator) + separator.length());
        for (int i = 0; i < argumentNames.size() && matching; i++) {
            if (!simplifiedName.startsWith(argumentNames.get(i))) {
                matching = false;
            }
            if (matching) {
                // removing found prefix column
                simplifiedName = simplifiedName.replaceFirst(argumentNames.get(i), "");
                // in case of we have more fields (concatenated with And, or with Or) we are removing these
                // This part serves to remove modifiers like: findAllByNameContainingIgnoreCaseAndAge
                if (simplifiedName.contains("And")) {
                    simplifiedName = simplifiedName.substring(simplifiedName.indexOf("And") + 3);
                } else if (simplifiedName.contains("Or")) {
                    simplifiedName = simplifiedName.substring(simplifiedName.indexOf("Or") + 2);
                }
            }
        }
        return matching;
    }

    private MethodInfo createMethodInfo(JpaRepository<?, ?> repository, String findMethodName, String countMethodName, List<Class<?>> argumentTypes) {
        argumentTypes.remove(Pageable.class);
        return MethodInfo.builder()
                .repository(repository)
                .findMethodName(findMethodName)
                .countMethodName(countMethodName)
                .argumentTypes(argumentTypes)
                .build();
    }
}
