package io.overcoded.grid.processor;

import io.overcoded.grid.MenuLayout;
import io.overcoded.grid.annotation.GridSystem;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.reflections.Reflections;

import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;

/**
 * MenuLayoutFactory creates a MenuLayout based on GridSystem, if present.
 */
@Slf4j
@RequiredArgsConstructor
public class MenuLayoutFactory {
    /**
     * Internal cache for MenuLayout related to type
     */
    private final Map<Class<?>, MenuLayout> typeCache = new ConcurrentHashMap<>();
    private final ReflectionsFactory reflectionsFactory;
    private final MenuGroupCollector menuGroupCollector;

    /**
     * Looks after GridSystem annotation starting from the base package.
     * If found, converting it to MenuLayout.
     *
     * @return MenuLayout of the GridSystem if present
     */
    public Optional<MenuLayout> create() {
        Reflections reflections = reflectionsFactory.create();
        return reflections.getTypesAnnotatedWith(GridSystem.class)
                .stream()
                .findFirst()
                .map(this::convert);
    }

    /**
     * Converts the package annotated with GridSystem to MenuLayout.
     * Based on this package we are looking for menus, views and we're
     * organizing them.
     *
     * @param type package annotated with GridSystem
     * @return MenuLayout containing all the menu groups and entries
     */
    private MenuLayout convert(Class<?> type) {
        return typeCache.computeIfAbsent(type, this::createMenuLayout);
    }

    private MenuLayout createMenuLayout(Class<?> type) {
        GridSystem gridSystem = type.getAnnotation(GridSystem.class);
        log.debug("Grid system found for {}", type.getName());
        MenuLayout result = MenuLayout.builder()
                .type(type)
                .icon(gridSystem.icon())
                .title(gridSystem.title())
                .groups(menuGroupCollector.collect(type.getPackageName(), gridSystem.path()))
                .build();
        log.trace("Converted menu layout: {}", result);
        return result;
    }
}
