package io.overcoded.grid.processor;

import io.overcoded.grid.MenuGroup;
import io.overcoded.grid.annotation.GridMenu;
import io.overcoded.grid.processor.column.NameTransformer;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.util.List;

/**
 * Creates MenuGroup from type annotated with GridMenu
 */
@Slf4j
@RequiredArgsConstructor
public class MenuGroupFactory {
    private final MenuEntryCollector menuEntryCollector;
    private final NameTransformer nameTransformer;

    /**
     * Creates a MenuGroup entry from a package.
     * The MenuGroup also collects the menu entries belong to the group.
     * <p>
     * Each menu group will be prefixed with the menu group's path.
     *
     * @param type       package which should be annotated with GridMenu
     * @param pathPrefix path prefix of the system.
     * @return a MenuGroup created based on MenuGroup
     * @throws IllegalArgumentException when GridMenu annotation not present
     */
    public MenuGroup create(Class<?> type, String pathPrefix) {
        if (!type.isAnnotationPresent(GridMenu.class)) {
            throw new IllegalArgumentException("Input class should be annotated with GridMenu.");
        }
        log.debug("Creating menu group for {} under {}", type.getName(), pathPrefix);
        GridMenu source = type.getAnnotation(GridMenu.class);
        return MenuGroup
                .builder()
                .type(type)
                .icon(source.icon())
                .order(source.order())
                .label(getLabel(source))
                .path(pathPrefix + source.path())
                .enabledFor(List.of(source.enabledFor()))
                .entries(menuEntryCollector.collect(type.getPackageName(), pathPrefix + source.path()))
                .build();
    }

    private String getLabel(GridMenu source) {
        return source.label().isBlank()
                ? nameTransformer.transformPath(getFinalPathSegment(source))
                : source.label();
    }

    private String getFinalPathSegment(GridMenu source) {
        return source.path().substring(source.path().lastIndexOf("/") + 1);
    }
}
