package io.overcoded.grid.processor;

import io.overcoded.grid.MenuGroup;
import io.overcoded.grid.annotation.GridMenu;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.reflections.Reflections;

import java.util.List;
import java.util.Objects;

import static java.util.stream.Collectors.toList;

/**
 * Collects the MenuGroups in the system
 */
@Slf4j
@RequiredArgsConstructor
public class MenuGroupCollector {
    private final ReflectionsFactory reflectionsFactory;
    private final MenuGroupFactory menuGroupFactory;

    /**
     * Looking after packages annotated with GridMenu under the base package.
     *
     * @param basePackage package name where GridMenus should be searched
     * @param pathPrefix  path prefix of the system (extracted from GridSystem)
     * @return List of available MenuGroups in the specified package
     */
    public List<MenuGroup> collect(String basePackage, String pathPrefix) {
        log.debug("Collecting menu groups for {} under {}", basePackage, pathPrefix);
        Reflections reflections = reflectionsFactory.create(basePackage);
        List<MenuGroup> result = reflections.getTypesAnnotatedWith(GridMenu.class)
                .stream()
                .map(type -> menuGroupFactory.create(type, pathPrefix))
                .filter(Objects::nonNull)
                .sorted()
                .collect(toList());
        log.trace("Collected menu groups: {}", result);
        return result;
    }
}
