package io.overcoded.grid.processor;

import io.overcoded.grid.MenuEntry;
import io.overcoded.grid.annotation.GridView;
import io.overcoded.grid.processor.column.NameTransformer;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.List;

/**
 * Creates a MenuEntry from a type, annotated with GridView.
 */
@Slf4j
@RequiredArgsConstructor
public class MenuEntryFactory {
    private final NameTransformer nameTransformer;

    /**
     * Creates a MenuEntry from a type.
     *
     * @param type       which should be processed and annotated with GridView
     * @param pathPrefix prefix of the enclosed MenuGroup.
     * @return MenuEntry created from GridView.
     * @throws IllegalArgumentException in case when GridView annotation is not present.
     */
    public MenuEntry create(Class<?> type, String pathPrefix) {
        if (!type.isAnnotationPresent(GridView.class)) {
            throw new IllegalArgumentException("Input class should be annotated with GridView.");
        }
        log.debug("Creating menu entry for {} under {}", type.getName(), pathPrefix);
        GridView source = type.getAnnotation(GridView.class);
        return MenuEntry
                .builder()
                .type(type)
                .icon(source.icon())
                .order(source.order())
                .entries(new ArrayList<>())
                .grouped(source.grouping())
                .path(pathPrefix + source.path())
                .label(nameTransformer.transformPath(getFinalPathSegment(source)))
                .enabledFor(List.of(source.enabledFor()))
                .build();
    }

    private String getFinalPathSegment(GridView source) {
        return source.path().substring(source.path().lastIndexOf("/") + 1);
    }
}
