package io.overcoded.grid.processor;

import io.overcoded.grid.MenuEntry;
import io.overcoded.grid.annotation.GridView;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.reflections.Reflections;

import java.util.List;
import java.util.Objects;

import static java.util.function.Predicate.not;
import static java.util.stream.Collectors.toList;

/**
 * Collects all the MenuEntry in a package.
 */
@Slf4j
@RequiredArgsConstructor
public class MenuEntryCollector {
    private final MenuEntryFactory menuEntryFactory;
    private final ReflectionsFactory reflectionsFactory;
    private final SubmenuEntryCollector submenuEntryCollector;

    /**
     * Collects all the GridView's under the specified package.
     * Convert them into MenuEntry instances and rearrange them
     * based on grouping.
     *
     * @param basePackage package where GridView should be searched
     * @param pathPrefix prefix of the enclosed MenuGroup.
     * @return List of MenuEntries
     */
    public List<MenuEntry> collect(String basePackage, String pathPrefix) {
        log.debug("Collecting menu entries in {} for {}", basePackage, pathPrefix);
        Reflections reflections = reflectionsFactory.create(basePackage);
        List<MenuEntry> views = collectAllMenuEntries(pathPrefix, reflections);
        List<MenuEntry> submenuEntries = collectSubmenus(views);

        List<MenuEntry> result = views.stream()
                .filter(not(MenuEntry::isGrouped))
                .map(menuEntry -> extendWithSubmenus(menuEntry, submenuEntries))
                .sorted()
                .collect(toList());
        log.trace("Collected menu entries: {}", result);
        return result;
    }

    private static List<MenuEntry> collectSubmenus(List<MenuEntry> views) {
        return views.stream()
                .filter(MenuEntry::isGrouped)
                .collect(toList());
    }

    private List<MenuEntry> collectAllMenuEntries(String pathPrefix, Reflections reflections) {
        return reflections.getTypesAnnotatedWith(GridView.class)
                .stream()
                .map(type -> menuEntryFactory.create(type, pathPrefix))
                .filter(Objects::nonNull)
                .collect(toList());
    }

    private MenuEntry extendWithSubmenus(MenuEntry menuEntry, List<MenuEntry> submenuEntries) {
        menuEntry.setEntries(submenuEntryCollector.collect(menuEntry, submenuEntries));
        return menuEntry;
    }
}
