package io.overcoded.grid.processor;

import io.overcoded.grid.ColumnInfo;
import io.overcoded.grid.ContextMenuGroup;
import io.overcoded.grid.GridInfo;
import io.overcoded.grid.MethodInfo;
import io.overcoded.grid.annotation.GridDialog;
import io.overcoded.grid.processor.column.NameTransformer;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Creates a GridInfo from type which is annotated with GridDialog.
 */
@Slf4j
@RequiredArgsConstructor
public class GridInfoDialogFactory implements GridInfoFactory<GridDialog> {
    /**
     * Internal cache for GridInfo related to type
     */
    private final Map<Class<?>, GridInfo> typeCache = new ConcurrentHashMap<>();
    private final NameTransformer nameTransformer;
    private final MethodInfoFactory methodInfoFactory;
    private final ColumnInfoCollector columnInfoCollector;
    private final ContextMenuGroupCollector contextMenuGroupCollector;

    /**
     * Corresponding annotation type. As this implementation based on
     * GridDialog annotation, this method should return GridDialog.class.
     *
     * @return expected annotation on input type of create method.
     */
    @Override
    public Class<GridDialog> getSourceType() {
        return GridDialog.class;
    }

    /**
     * Creates a GridInfo instance based on the input type GridDialog annotation.
     * It also sets the columns and groups for the GridInfo.
     *
     * @param type the type which should be transformed into GridInfo
     * @return a GridInfo instance
     * @throws IllegalArgumentException when type is not annotated with GridDialog
     */
    public GridInfo create(Class<?> type) {
        if (!type.isAnnotationPresent(getSourceType())) {
            throw new IllegalArgumentException("GridDialog annotation must present on input type.");
        }
        return typeCache.computeIfAbsent(type, this::createGridInfo);
    }

    private GridInfo createGridInfo(Class<?> type) {
        log.debug("Creating grid info for {}", type.getName());
        GridDialog source = type.getAnnotation(getSourceType());
        List<ColumnInfo> columns = columnInfoCollector.collect(type);
        List<ContextMenuGroup> contextMenuGroups = contextMenuGroupCollector.collect(type);
        MethodInfo methodInfo = methodInfoFactory.create(type, columns);
        String name = nameTransformer.transform(type.getSimpleName());
        return GridInfo.builder()
                .enabledMethods(List.of(source.enabledMethods()))
                .enabledFor(List.of(source.enabledFor()))
                .contextMenuGroups(contextMenuGroups)
                .sourceAnnotation(getSourceType())
                .description(source.description())
                .methodInfo(methodInfo)
                .columns(columns)
                .type(type)
                .name(name)
                .build();
    }
}
