package io.overcoded.grid.processor;

import io.overcoded.grid.ContextMenuEntry;
import io.overcoded.grid.annotation.GridDialog;
import io.overcoded.grid.processor.column.JoinFieldFinder;
import io.overcoded.grid.processor.column.NameTransformer;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.Objects;

/**
 * ContextMenuEntryFactory creates ContextMenuEntry from a field.
 */
@Slf4j
@RequiredArgsConstructor
public class ContextMenuEntryFactory {
    private final JoinFieldFinder joinFieldFinder;
    private final NameTransformer nameTransformer;

    /**
     * Creates a ContextMenuEntry based on a Field. The type belong to the Field should be annotated with GridDialog.
     *
     * @param field       field which should be converted into ContextMenuEntry
     * @param genericType as context menu mostly coming from a generic collection, this type is the generic type of the collection
     * @param parentType  type of the
     * @return a ContextMenuEntry if GridDialog is present on the type of the field, null otherwise
     */
    public ContextMenuEntry create(Field field, Class<?> genericType, Class<?> parentType) {
        ContextMenuEntry columnInfo = null;
        log.debug("Creating context menu entry for {} in {}", field.getName(), parentType.getName());
        if (genericType.isAnnotationPresent(GridDialog.class)) {
            GridDialog gridDialog = genericType.getAnnotation(GridDialog.class);
            Field joinField = joinFieldFinder.find(genericType, parentType);
            String joinFieldName = Objects.nonNull(joinField) ? joinField.getName() : null;
            columnInfo = ContextMenuEntry
                    .builder()
                    .type(genericType)
                    .name(joinFieldName)
                    .parentType(parentType)
                    .icon(gridDialog.icon())
                    .order(gridDialog.order())
                    .parentFieldName(field.getName())
                    .description(gridDialog.description())
                    .label(nameTransformer.transform(field))
                    .enabledFor(Arrays.asList(gridDialog.enabledFor()))
                    .build();
        }
        return columnInfo;
    }
}
