package io.overcoded.grid;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

import java.util.List;

/**
 * A menu entry which opens a dialog
 */
@Data
@Builder
@NoArgsConstructor
@AllArgsConstructor
public class ContextMenuEntry implements Comparable<ContextMenuEntry> {
    /**
     * Type of the dialog.
     * <p>
     * Originally we designed this to use with Vaadin, where there is
     * a Dialog class, but something like this may exist for other
     * frameworks.
     * <p>
     * So if the type is an assignable form of "this" Dialog class,
     * then it should be displayed.
     * <p>
     * If this is an entity type annotated with GridDialog or GridView
     * Which means the grid inside this dialog should handle this type.
     * <p>
     * In both cases the dialog should accept parent type as parameter.
     */
    private Class<?> type;

    /**
     * Name of the field which should be filtered in the dialog.
     */
    private String name;

    /**
     * Name of the field in the parent class
     */
    private String parentFieldName;

    /**
     * Parent type defines the type of the view (or dialog) which
     * contains this dialog.
     * <p>
     * When displaying, the dialog should get an instance of this
     * type.
     */
    private Class<?> parentType;

    /**
     * Icon, which should be displayed in the context menu
     */
    private String icon;

    /**
     * Custom label for the menu entry.
     * If it's not present, the type name will be used.
     */
    private String label;

    /**
     * Url of the page which should be opened
     * <p>
     * This is an extension for context menu.
     */
    private String url;

    /**
     * Description or information to display as a warning on the
     * top section of the dialog.
     */
    private String description;

    /**
     * The position of the menu item in the group.
     */
    private int order;

    /**
     * List of roles who can access this dialog.
     * <p>
     * Implementation can be different: the menu item can be disabled
     * or not even visible.
     */
    private List<String> enabledFor;

    @Override
    public int compareTo(ContextMenuEntry other) {
        return Integer.compare(order, other.order);
    }
}
