package io.overcoded.repository.annotation.processor;

import io.overcoded.repository.annotation.processor.domain.MethodData;
import io.overcoded.repository.annotation.processor.domain.RepositoryData;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.VelocityEngine;
import org.apache.velocity.runtime.RuntimeConstants;
import org.apache.velocity.runtime.resource.loader.ClasspathResourceLoader;

import java.io.StringWriter;

/**
 * @author Diana Ladanyi
 */
public class RepositoryGenerator {
    private static final String REPOSITORY_TEMPLATE_FILE = "templates/repository_template.vm";
    private static final String COUNT_METHOD_TEMPLATE_FILE = "templates/count_method_template.vm";
    private static final String FIND_METHOD_TEMPLATE_FILE = "templates/find_method_template.vm";
    private static final String ENTITY_TYPE_TEMPLATE_VARIABLE = "entityType";
    private static final String PACKAGE_NAME_TEMPLATE_VARIABLE = "packageName";
    private static final String SIMPLE_CLASS_NAME_TEMPLATE_VARIABLE = "simpleClassName";
    private static final String METHODS_TEMPLATE_VARIABLE = "methods";
    private static final String METHOD_NAME_TEMPLATE_VARIABLE = "methodName";
    private static final String PARAMETERS_TEMPLATE_VARIABLE = "parameters";

    public String generateRepository(RepositoryData repositoryData) {
        VelocityEngine velocityEngine = createEngine();
        Template template = velocityEngine.getTemplate(REPOSITORY_TEMPLATE_FILE);
        VelocityContext context = createContextForRepository(repositoryData, velocityEngine);
        StringWriter writer = new StringWriter();
        template.merge(context, writer);
        return writer.toString();
    }

    private String generateMethods(VelocityEngine velocityEngine, RepositoryData repositoryData) {
        StringWriter writer = new StringWriter();
        createFindMethods(velocityEngine, repositoryData, writer);
        createCountMethods(velocityEngine, repositoryData, writer);
        return writer.toString();
    }

    private void createCountMethods(VelocityEngine velocityEngine, RepositoryData repositoryData, StringWriter writer) {
        repositoryData.getCountMethods().forEach(methodData -> createMethod(velocityEngine, repositoryData, writer, methodData, COUNT_METHOD_TEMPLATE_FILE));
    }

    private void createFindMethods(VelocityEngine velocityEngine, RepositoryData repositoryData, StringWriter writer) {
        repositoryData.getFindMethods().forEach(methodData -> createMethod(velocityEngine, repositoryData, writer, methodData, FIND_METHOD_TEMPLATE_FILE));
    }

    private void createMethod(VelocityEngine velocityEngine, RepositoryData repositoryData, StringWriter writer, MethodData methodData, String findMethodTemplateFile) {
        Template template = velocityEngine.getTemplate(findMethodTemplateFile);
        VelocityContext context = createContextForMethod(repositoryData, methodData);
        template.merge(context, writer);
    }

    private VelocityEngine createEngine() {
        VelocityEngine velocityEngine = new VelocityEngine();
        velocityEngine.setProperty(RuntimeConstants.RESOURCE_LOADER, "classpath");
        velocityEngine.setProperty("classpath.resource.loader.class", ClasspathResourceLoader.class.getName());
        velocityEngine.init();
        return velocityEngine;
    }

    private VelocityContext createContextForRepository(RepositoryData repositoryData, VelocityEngine velocityEngine) {
        VelocityContext context = new VelocityContext();
        context.put(ENTITY_TYPE_TEMPLATE_VARIABLE, repositoryData.getEntityType());
        context.put(PACKAGE_NAME_TEMPLATE_VARIABLE, repositoryData.getPackageName());
        context.put(SIMPLE_CLASS_NAME_TEMPLATE_VARIABLE, repositoryData.getSimpleClassName());
        context.put(METHODS_TEMPLATE_VARIABLE, generateMethods(velocityEngine, repositoryData));
        return context;
    }

    private VelocityContext createContextForMethod(RepositoryData repositoryData, MethodData methodData) {
        VelocityContext context = new VelocityContext();
        context.put(ENTITY_TYPE_TEMPLATE_VARIABLE, repositoryData.getEntityType());
        context.put(METHOD_NAME_TEMPLATE_VARIABLE, methodData.getMethodName());
        context.put(PARAMETERS_TEMPLATE_VARIABLE, methodData.getParameters());
        return context;
    }
}
