package io.overcoded.repository.annotation.processor;

import com.google.auto.service.AutoService;
import io.overcoded.repository.annotation.processor.collector.ElementCollector;
import io.overcoded.repository.annotation.processor.collector.ElementCollectorFactory;
import io.overcoded.repository.annotation.processor.domain.ElementCollection;
import io.overcoded.repository.annotation.processor.domain.RepositoryData;
import io.overcoded.repository.annotation.processor.supplier.SupplierManager;
import io.overcoded.repository.annotation.processor.supplier.SupplierManagerFactory;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import javax.annotation.processing.*;
import javax.lang.model.SourceVersion;
import javax.lang.model.element.TypeElement;
import javax.tools.JavaFileObject;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

/**
 * @author Diana Ladanyi
 */
@Slf4j
@RequiredArgsConstructor
@AutoService(Processor.class)
@SupportedSourceVersion(SourceVersion.RELEASE_11)
@SupportedAnnotationTypes(value = {
        "io.overcoded.repository.annotation.FindAll",
        "io.overcoded.repository.annotation.FindAllBy",
        "io.overcoded.repository.annotation.FindAllArray",
        "io.overcoded.repository.annotation.DynamicRepository"})
public class DynamicRepositoryProcessor extends AbstractProcessor {
    private static final String REPOSITORY_CLASS_SUFFIX = "JpaRepository";
    private static final String PACKAGE_SEPARATOR = ".";
    private final SupplierManager supplierManager;
    private final RepositoryGenerator repositoryGenerator;
    private final Map<String, ElementCollector> fillers;

    public DynamicRepositoryProcessor() {
        this(SupplierManagerFactory.createDefaultManager(), new RepositoryGenerator(), ElementCollectorFactory.createElementCollectorMap());
    }

    @Override
    public boolean process(Set<? extends TypeElement> annotations, RoundEnvironment roundEnv) {
        Map<String, ElementCollection> elementCollectionMap = getElementCollectionMap(annotations, roundEnv);
        evaluateElementCollectionMap(elementCollectionMap);
        return false;
    }

    private Map<String, ElementCollection> getElementCollectionMap(Set<? extends TypeElement> annotations, RoundEnvironment roundEnv) {
        Map<String, ElementCollection> elementCollectionMap = new HashMap<>();
        for (TypeElement annotation : annotations) {
            String annotationName = annotation.getSimpleName().toString();
            if (fillers.containsKey(annotationName)) {
                fillers.get(annotationName).accept(elementCollectionMap, roundEnv.getElementsAnnotatedWith(annotation));
            }
        }
        return elementCollectionMap;
    }

    private void evaluateElementCollectionMap(Map<String, ElementCollection> elementCollectionMap) {
        elementCollectionMap.forEach((entityType, elementCollection) -> {
            if (Objects.nonNull(elementCollection.getDynamicRepositoryElement())) {
                RepositoryData repositoryData = supplierManager.requestSupply(elementCollection, new RepositoryData());
                try {
                    writeBuilderFile(repositoryData);
                } catch (IOException e) {
                    log.error("An exception occurred: {}", e.getMessage());
                }
            }
        });
    }

    private void writeBuilderFile(RepositoryData repositoryData) throws IOException {
        String simpleClassName = repositoryData.getEntityType() + REPOSITORY_CLASS_SUFFIX;
        String builderClassName = repositoryData.getPackageName() + PACKAGE_SEPARATOR + simpleClassName;

        repositoryData.setSimpleClassName(simpleClassName);
        String file = repositoryGenerator.generateRepository(repositoryData);

        JavaFileObject builderFile = processingEnv.getFiler().createSourceFile(builderClassName);

        try (PrintWriter out = new PrintWriter(builderFile.openWriter())) {
            out.println(file);
        }
    }

}
