package io.overcoded.vaadin;

import com.vaadin.flow.component.contextmenu.MenuItem;
import com.vaadin.flow.component.contextmenu.SubMenu;
import com.vaadin.flow.component.contextmenu.SubMenuBase;
import com.vaadin.flow.component.dialog.Dialog;
import com.vaadin.flow.component.grid.ColumnTextAlign;
import com.vaadin.flow.component.grid.Grid;
import com.vaadin.flow.component.icon.Icon;
import com.vaadin.flow.component.icon.VaadinIcon;
import com.vaadin.flow.component.menubar.MenuBar;
import com.vaadin.flow.component.menubar.MenuBarVariant;
import com.vaadin.flow.spring.annotation.SpringComponent;
import io.overcoded.grid.ContextMenuEntry;
import io.overcoded.grid.ContextMenuGroup;
import io.overcoded.grid.GridInfo;
import io.overcoded.grid.security.GridSecurityService;
import io.overcoded.vaadin.dialog.DynamicDialogFactory;
import io.overcoded.vaadin.dialog.DynamicDialogParameter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.vaadin.crudui.crud.impl.GridCrud;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;

@Slf4j
@SpringComponent
@RequiredArgsConstructor
public class ContextMenuConfigurer {
    private final DynamicDialogFactory dynamicDialogFactory;
    private final GridSecurityService securityService;

    <T> void configure(GridCrud<T> gridCrud, GridInfo gridInfo) {
        List<ContextMenuGroup> menuGroups = getEnabledMenuGroups(gridInfo);
        Grid<T> grid = gridCrud.getGrid();
        if (!menuGroups.isEmpty()) {
            grid.addComponentColumn(item -> createMenuBar(gridCrud, menuGroups, item))
                    .setSortable(false)
                    .setAutoWidth(true)
                    .setTextAlign(ColumnTextAlign.END)
                    .setFrozenToEnd(true);
        }
    }

    private List<ContextMenuGroup> getEnabledMenuGroups(GridInfo gridInfo) {
        return gridInfo.getContextMenuGroups()
                .stream()
                .filter(group -> !getEnabledMenuEntries(group.getContextMenuEntries()).isEmpty())
                .toList();
    }

    private List<ContextMenuEntry> getEnabledMenuEntries(List<ContextMenuEntry> menuEntries) {
        return menuEntries.stream()
                .filter(entry -> securityService.hasPermission(entry.getEnabledFor()))
                .toList();
    }

    private <T> MenuBar createMenuBar(GridCrud<T> gridCrud, List<ContextMenuGroup> menuGroups, T item) {
        MenuBar menuBar = new MenuBar();
        menuBar.addThemeVariants(MenuBarVariant.LUMO_TERTIARY);
        menuBar.addThemeVariants(MenuBarVariant.LUMO_ICON);
        addMenus(gridCrud, menuGroups, item, menuBar.addItem("•••"));
        return menuBar;
    }

    private <T> void addMenus(GridCrud<T> gridCrud, List<ContextMenuGroup> menuGroups, T item, MenuItem menuItem) {
        if (menuGroups.size() > 1) {
            addMenuGroups(gridCrud, item, menuItem, menuGroups);
        } else {
            addMenuEntries(gridCrud, item, menuItem, menuGroups.get(0).getContextMenuEntries());
        }
    }

    private <T> void addMenuGroups(GridCrud<T> gridCrud, T item, MenuItem primaryMenuItem, List<ContextMenuGroup> menuGroups) {
        SubMenu primarySubMenu = primaryMenuItem.getSubMenu();
        menuGroups.forEach(menuGroup -> {
            MenuItem menuItem = primarySubMenu.addItem(menuGroup.getLabel());
            findIcon(menuGroup.getIcon()).ifPresent(menuItem::addComponentAsFirst);
            addMenuEntries(gridCrud, item, menuItem, menuGroup.getContextMenuEntries());
            if (!hasMenuItem(menuItem)) {
                primarySubMenu.remove(menuItem);
            }
        });
    }

    private boolean hasMenuItem(MenuItem menuItem) {
        return !Optional.ofNullable(menuItem.getSubMenu()).map(SubMenuBase::getItems).orElseGet(List::of).isEmpty();
    }

    private <T, U> void addMenuEntries(GridCrud<U> gridCrud, U item, MenuItem menuItem, List<ContextMenuEntry> menuEntries) {
        if (!menuEntries.isEmpty()) {
            SubMenu subMenu = menuItem.getSubMenu();
            menuEntries.stream()
                    .filter(entry -> securityService.hasPermission(entry.getEnabledFor()))
                    .forEach(entry -> {
                        MenuItem subMenuItem = subMenu.addItem(entry.getLabel());
                        findChildIcon(entry.getIcon()).ifPresent(subMenuItem::addComponentAsFirst);
                        subMenuItem.addClickListener(event -> {
                            DynamicDialogParameter<T, U> parameter = getDialogParameter(item, entry);
                            Dialog dialog = dynamicDialogFactory.create(gridCrud, parameter, this::configure);
                            dialog.open();
                            log.info("Clicked item: {}", item);
                        });
                    });
        }
    }

    private <T, U> DynamicDialogParameter<T, U> getDialogParameter(U item, ContextMenuEntry entry) {
        return DynamicDialogParameter
                .<T, U>builder()
                .type((Class<T>) entry.getType())
                .property(entry.getName())
                .parameter(item)
                .entry(entry)
                .build();
    }

    private Optional<Icon> findChildIcon(String icon) {
        return findIcon(icon).map(vaadinIcon -> {
            vaadinIcon.getStyle().set("width", "var(--lumo-icon-size-s)");
            vaadinIcon.getStyle().set("height", "var(--lumo-icon-size-s)");
            vaadinIcon.getStyle().set("marginRight", "var(--lumo-space-s)");
            return vaadinIcon;
        });
    }

    private Optional<Icon> findIcon(String icon) {
        return Arrays.stream(VaadinIcon.values())
                .filter(vaadinIcon -> vaadinIcon.name().equalsIgnoreCase(icon))
                .map(VaadinIcon::create)
                .findFirst();
    }


}
