/*
 * Decompiled with CFR 0.152.
 */
package io.opencmw.utils;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;
import org.jetbrains.annotations.NotNull;

public class Cache<K, V>
implements Map<K, V> {
    private final ConcurrentHashMap<K, V> dataCache = new ConcurrentHashMap();
    private final ConcurrentHashMap<K, Instant> timeOutMap = new ConcurrentHashMap();
    private final ChronoUnit chronoUnit;
    private final TimeUnit timeUnit;
    private final long timeOut;
    private final int limit;
    private final BiConsumer<K, V> preListener;
    private final BiConsumer<K, V> postListener;

    public Cache(int limit) {
        this(0L, TimeUnit.MILLISECONDS, limit, null, null);
    }

    public Cache(long timeOut, TimeUnit timeUnit) {
        this(timeOut, timeUnit, Integer.MAX_VALUE, null, null);
    }

    public Cache(long timeOut, TimeUnit timeUnit, int limit) {
        this(timeOut, timeUnit, limit, null, null);
    }

    private Cache(long timeOut, TimeUnit timeUnit, int limit, BiConsumer<K, V> preListener, BiConsumer<K, V> postListener) {
        if (timeOut < 0L) {
            throw new IllegalArgumentException("Timeout cannot be negative");
        }
        if (timeOut > 0L && null == timeUnit) {
            throw new IllegalArgumentException("TimeUnit cannot be null if timeOut is > 0");
        }
        if (limit < 1) {
            throw new IllegalArgumentException("Limit cannot be smaller than 1");
        }
        this.timeOut = timeOut;
        this.timeUnit = timeUnit;
        this.chronoUnit = Cache.convertToChronoUnit(timeUnit);
        this.limit = limit;
        this.preListener = preListener;
        this.postListener = postListener;
        ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread t = Executors.defaultThreadFactory().newThread(r);
            t.setName(Cache.class.getCanonicalName() + "-Thread");
            t.setDaemon(true);
            return t;
        });
        if (timeOut != 0L) {
            executor.scheduleAtFixedRate(this::checkTime, 0L, timeOut, timeUnit);
        }
    }

    @Override
    public void clear() {
        this.dataCache.clear();
        this.timeOutMap.clear();
    }

    @Override
    public boolean containsKey(Object key) {
        return this.dataCache.containsKey(key);
    }

    @Override
    public boolean containsValue(Object value) {
        return this.dataCache.containsValue(value);
    }

    @Override
    @NotNull
    public Set<Map.Entry<K, V>> entrySet() {
        return this.dataCache.entrySet();
    }

    @Override
    public V get(Object key) {
        return this.getIfPresent(key);
    }

    public V getIfPresent(K key) {
        this.timeOutMap.put(key, Instant.now());
        return this.dataCache.getOrDefault(key, null);
    }

    public long getLimit() {
        return this.limit;
    }

    public Optional<V> getOptional(K key) {
        return Optional.ofNullable(this.getIfPresent(key));
    }

    public int getSize() {
        return this.dataCache.size();
    }

    public long getTimeout() {
        return this.timeOut;
    }

    public TimeUnit getTimeUnit() {
        return this.timeUnit;
    }

    @Override
    public boolean isEmpty() {
        return this.dataCache.isEmpty();
    }

    @Override
    @NotNull
    public Set<K> keySet() {
        return this.dataCache.keySet();
    }

    @Override
    public V put(K key, V value) {
        this.checkSize();
        V val = this.dataCache.put(key, value);
        this.timeOutMap.put(key, Instant.now());
        return val;
    }

    @Override
    public V putIfAbsent(K key, V value) {
        this.checkSize();
        V val = this.dataCache.putIfAbsent(key, value);
        this.timeOutMap.putIfAbsent(key, Instant.now());
        return val;
    }

    @Override
    public void putAll(Map<? extends K, ? extends V> m) {
        this.checkSize(m.size());
        this.dataCache.putAll(m);
        Instant now = Instant.now();
        m.keySet().forEach((? super T key) -> this.timeOutMap.putIfAbsent(key, now));
    }

    @Override
    public V remove(Object key) {
        V val = this.dataCache.remove(key);
        this.timeOutMap.remove(key);
        return val;
    }

    @Override
    public int size() {
        return this.dataCache.size();
    }

    @Override
    @NotNull
    public Collection<V> values() {
        return this.dataCache.values();
    }

    protected void checkSize() {
        this.checkSize(1);
    }

    protected void checkSize(int nNewElements) {
        if (this.dataCache.size() < this.limit) {
            return;
        }
        int surplusEntries = Math.max(this.dataCache.size() - this.limit + nNewElements, 0);
        List toBeRemoved = this.timeOutMap.entrySet().stream().sorted(Map.Entry.comparingByValue().reversed()).limit(surplusEntries).map(Map.Entry::getKey).collect(Collectors.toList());
        this.removeEntries(toBeRemoved);
    }

    protected void checkTime() {
        Instant cutoffTime = Instant.now().minus(this.timeOut, this.chronoUnit);
        List toBeRemoved = this.timeOutMap.entrySet().stream().filter(entry -> ((Instant)entry.getValue()).isBefore(cutoffTime)).map(Map.Entry::getKey).collect(Collectors.toList());
        this.removeEntries(toBeRemoved);
    }

    private void removeEntries(List<K> toBeRemoved) {
        HashMap<K, V> removalMap;
        if (this.preListener == null && this.postListener == null) {
            removalMap = null;
        } else {
            removalMap = new HashMap<K, V>();
            toBeRemoved.forEach((? super T key) -> removalMap.put(key, this.dataCache.get(key)));
        }
        if (this.preListener != null) {
            removalMap.forEach(this.preListener);
        }
        toBeRemoved.forEach((? super T key) -> {
            this.timeOutMap.remove(key);
            this.dataCache.remove(key);
        });
        if (this.postListener != null) {
            removalMap.forEach(this.postListener);
        }
    }

    public static <K3, V3> CacheBuilder<K3, V3> builder() {
        return new CacheBuilder();
    }

    protected static int clamp(int min, int max, int value) {
        if (value < min) {
            return min;
        }
        return Math.min(value, max);
    }

    protected static long clamp(long min, long max, long value) {
        if (value < min) {
            return min;
        }
        return Math.min(value, max);
    }

    protected static ChronoUnit convertToChronoUnit(TimeUnit timeUnit) {
        switch (timeUnit) {
            case NANOSECONDS: {
                return ChronoUnit.NANOS;
            }
            case MICROSECONDS: {
                return ChronoUnit.MICROS;
            }
            case SECONDS: {
                return ChronoUnit.SECONDS;
            }
            case MINUTES: {
                return ChronoUnit.MINUTES;
            }
            case HOURS: {
                return ChronoUnit.HOURS;
            }
            case DAYS: {
                return ChronoUnit.DAYS;
            }
        }
        return ChronoUnit.MILLIS;
    }

    public static class CacheBuilder<K2, V2> {
        private int limit = Integer.MAX_VALUE;
        private long timeOut;
        private TimeUnit timeUnit = TimeUnit.MILLISECONDS;
        private BiConsumer<K2, V2> preListener;
        private BiConsumer<K2, V2> postListener;

        private CacheBuilder() {
        }

        public Cache<K2, V2> build() {
            return new Cache<K2, V2>(this.timeOut, this.timeUnit, this.limit, this.preListener, this.postListener);
        }

        public CacheBuilder<K2, V2> withLimit(int limit) {
            if (limit < 1) {
                throw new IllegalArgumentException("Limit cannot be smaller than 1");
            }
            this.limit = limit;
            return this;
        }

        public CacheBuilder<K2, V2> withPostListener(BiConsumer<K2, V2> listener) {
            if (listener == null) {
                throw new IllegalArgumentException("listener cannot be null");
            }
            this.postListener = listener;
            return this;
        }

        public CacheBuilder<K2, V2> withPreListener(BiConsumer<K2, V2> listener) {
            if (listener == null) {
                throw new IllegalArgumentException("listener cannot be null");
            }
            this.preListener = listener;
            return this;
        }

        public CacheBuilder<K2, V2> withTimeout(long timeOut, TimeUnit timeUnit) {
            if (timeOut < 0L) {
                throw new IllegalArgumentException("Timeout cannot be negative");
            }
            if (null == timeUnit) {
                throw new IllegalArgumentException("TimeUnit cannot be null");
            }
            this.timeOut = Cache.clamp(0L, Integer.MAX_VALUE, timeOut);
            this.timeUnit = timeUnit;
            return this;
        }
    }
}

