package io.nlopez.loom.compose

import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.State
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.derivedStateOf
import androidx.compose.runtime.getValue
import androidx.compose.runtime.remember
import androidx.compose.ui.platform.LocalContext
import androidx.lifecycle.HasDefaultViewModelProviderFactory
import androidx.lifecycle.ViewModel
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.ViewModelStoreOwner
import androidx.lifecycle.viewmodel.compose.LocalViewModelStoreOwner
import io.nlopez.loom.LoomViewModel
import kotlinx.coroutines.flow.filterIsInstance
import kotlin.reflect.KProperty1

@Composable
public fun <VM : LoomViewModel<out Any, *>> loomViewModel(
  clazz: Class<VM>,
  named: String? = null,
  factory: ViewModelProvider.Factory? = LoomViewModelDefaults.viewModelProviderFactory
): VM {
  val storeOwner = checkNotNull(LocalViewModelStoreOwner.current) {
    "No ViewModelStoreOwner was provided via LocalViewModelStoreOwner"
  }
  return remember(storeOwner, clazz, named, factory) {
    storeOwner.get(clazz, named, factory)
  }
}

@Composable
public inline fun <reified VM : LoomViewModel<out Any, *>> loomViewModel(
  named: String? = null,
  factory: ViewModelProvider.Factory? = LoomViewModelDefaults.viewModelProviderFactory
): VM = loomViewModel(VM::class.java, named, factory)

// TODO move to its own loom-compose-hilt module
// @Composable
// inline fun <VS : Any, E, reified VM : LoomViewModel<VS, E>> loomViewModel(
//  entry: NavBackStackEntry,
//  named: String? = null
// ): VM = loomViewModel(VM::class.java, named, HiltViewModelFactory(LocalContext.current, entry))

public object LoomViewModelDefaults {
  public val viewModelProviderFactory: ViewModelProvider.Factory?
    @Composable get() {
      val currentContext = LocalContext.current
      return remember(currentContext) {
        (currentContext as? HasDefaultViewModelProviderFactory)?.defaultViewModelProviderFactory
      }
    }
}

private fun <T : ViewModel> ViewModelStoreOwner.get(
  javaClass: Class<T>,
  key: String? = null,
  factory: ViewModelProvider.Factory? = null
): T {
  val provider = if (factory != null) {
    ViewModelProvider(this, factory)
  } else {
    ViewModelProvider(this)
  }
  return if (key != null) {
    provider[key, javaClass]
  } else {
    provider[javaClass]
  }
}

@Composable
public inline fun <reified E : Any> LoomViewModel<*, E>.onEffects(
  crossinline function: suspend (effect: E) -> Unit
) {
  onEffect(function)
}

@Composable
public inline fun <reified E : Any> LoomViewModel<*, E>.onEffect(
  crossinline function: suspend (effect: E) -> Unit
) {
  LaunchedEffect(this) {
    effect.filterIsInstance<E>()
      .collect {
        function(it)
      }
  }
}

@Composable
public fun <VS : Any, E> LoomViewModel<VS, E>.collectAsState(): State<VS> = state.collectAsState()

@Composable
public fun <VS : Any, E, T1> LoomViewModel<VS, E>.collectAsStateWithMapper(
  mapper: (VS) -> T1
): State<T1> {
  val state by state.collectAsState()
  return remember { derivedStateOf { mapper(state) } }
}

@Composable
public fun <VS : Any, E, T1> LoomViewModel<VS, E>.collectAsState(
  property: KProperty1<VS, T1>
): State<T1> = collectAsStateWithMapper { property.get(it) }

@Composable
public fun <VS : Any, E, T1, T2> LoomViewModel<VS, E>.collectAsState(
  property1: KProperty1<VS, T1>,
  property2: KProperty1<VS, T2>
): State<Pair<T1, T2>> = collectAsStateWithMapper { it.toPair(property1, property2) }

private fun <VS : Any, T1, T2> VS.toPair(
  property1: KProperty1<VS, T1>,
  property2: KProperty1<VS, T2>
): Pair<T1, T2> = property1.get(this) to property2.get(this)

@Composable
public fun <VS : Any, E, T1, T2, T3> LoomViewModel<VS, E>.collectAsState(
  property1: KProperty1<VS, T1>,
  property2: KProperty1<VS, T2>,
  property3: KProperty1<VS, T3>,
): State<Tuple3<T1, T2, T3>> =
  collectAsStateWithMapper { it.toTuple3(property1, property2, property3) }

private fun <VS : Any, T1, T2, T3> VS.toTuple3(
  property1: KProperty1<VS, T1>,
  property2: KProperty1<VS, T2>,
  property3: KProperty1<VS, T3>
) = Tuple3(property1.get(this), property2.get(this), property3.get(this))

@Composable
public fun <VS : Any, E, T1, T2, T3, T4> LoomViewModel<VS, E>.collectAsState(
  property1: KProperty1<VS, T1>,
  property2: KProperty1<VS, T2>,
  property3: KProperty1<VS, T3>,
  property4: KProperty1<VS, T4>,
): State<Tuple4<T1, T2, T3, T4>> =
  collectAsStateWithMapper { it.toTuple4(property1, property2, property3, property4) }

private fun <VS : Any, T1, T2, T3, T4> VS.toTuple4(
  property1: KProperty1<VS, T1>,
  property2: KProperty1<VS, T2>,
  property3: KProperty1<VS, T3>,
  property4: KProperty1<VS, T4>
) = Tuple4(property1.get(this), property2.get(this), property3.get(this), property4.get(this))

@Composable
public fun <VS : Any, E, T1, T2, T3, T4, T5> LoomViewModel<VS, E>.collectAsState(
  property1: KProperty1<VS, T1>,
  property2: KProperty1<VS, T2>,
  property3: KProperty1<VS, T3>,
  property4: KProperty1<VS, T4>,
  property5: KProperty1<VS, T5>,
): State<Tuple5<T1, T2, T3, T4, T5>> =
  collectAsStateWithMapper { it.toTuple5(property1, property2, property3, property4, property5) }

private fun <VS : Any, T1, T2, T3, T4, T5> VS.toTuple5(
  property1: KProperty1<VS, T1>,
  property2: KProperty1<VS, T2>,
  property3: KProperty1<VS, T3>,
  property4: KProperty1<VS, T4>,
  property5: KProperty1<VS, T5>
) = Tuple5(
  property1.get(this),
  property2.get(this),
  property3.get(this),
  property4.get(this),
  property5.get(this)
)

@Composable
public fun <VS : Any, E, T1, T2, T3, T4, T5, T6> LoomViewModel<VS, E>.collectAsState(
  property1: KProperty1<VS, T1>,
  property2: KProperty1<VS, T2>,
  property3: KProperty1<VS, T3>,
  property4: KProperty1<VS, T4>,
  property5: KProperty1<VS, T5>,
  property6: KProperty1<VS, T6>,
): State<Tuple6<T1, T2, T3, T4, T5, T6>> =
  collectAsStateWithMapper {
    it.toTuple6(
      property1,
      property2,
      property3,
      property4,
      property5,
      property6
    )
  }

private fun <VS : Any, T1, T2, T3, T4, T5, T6> VS.toTuple6(
  property1: KProperty1<VS, T1>,
  property2: KProperty1<VS, T2>,
  property3: KProperty1<VS, T3>,
  property4: KProperty1<VS, T4>,
  property5: KProperty1<VS, T5>,
  property6: KProperty1<VS, T6>
) = Tuple6(
  property1.get(this),
  property2.get(this),
  property3.get(this),
  property4.get(this),
  property5.get(this),
  property6.get(this)
)

@Composable
public fun <VS : Any, E, T1, T2, T3, T4, T5, T6, T7> LoomViewModel<VS, E>.collectAsState(
  property1: KProperty1<VS, T1>,
  property2: KProperty1<VS, T2>,
  property3: KProperty1<VS, T3>,
  property4: KProperty1<VS, T4>,
  property5: KProperty1<VS, T5>,
  property6: KProperty1<VS, T6>,
  property7: KProperty1<VS, T7>,
): State<Tuple7<T1, T2, T3, T4, T5, T6, T7>> =
  collectAsStateWithMapper {
    it.toTuple7(
      property1,
      property2,
      property3,
      property4,
      property5,
      property6,
      property7
    )
  }

private fun <VS : Any, T1, T2, T3, T4, T5, T6, T7> VS.toTuple7(
  property1: KProperty1<VS, T1>,
  property2: KProperty1<VS, T2>,
  property3: KProperty1<VS, T3>,
  property4: KProperty1<VS, T4>,
  property5: KProperty1<VS, T5>,
  property6: KProperty1<VS, T6>,
  property7: KProperty1<VS, T7>,
) = Tuple7(
  property1.get(this),
  property2.get(this),
  property3.get(this),
  property4.get(this),
  property5.get(this),
  property6.get(this),
  property7.get(this)
)

@Composable
public fun <VS : Any, E, T1, T2, T3, T4, T5, T6, T7, T8> LoomViewModel<VS, E>.collectAsState(
  property1: KProperty1<VS, T1>,
  property2: KProperty1<VS, T2>,
  property3: KProperty1<VS, T3>,
  property4: KProperty1<VS, T4>,
  property5: KProperty1<VS, T5>,
  property6: KProperty1<VS, T6>,
  property7: KProperty1<VS, T7>,
  property8: KProperty1<VS, T8>,
): State<Tuple8<T1, T2, T3, T4, T5, T6, T7, T8>> =
  collectAsStateWithMapper {
    it.toTuple8(
      property1,
      property2,
      property3,
      property4,
      property5,
      property6,
      property7,
      property8
    )
  }

private fun <VS : Any, T1, T2, T3, T4, T5, T6, T7, T8> VS.toTuple8(
  property1: KProperty1<VS, T1>,
  property2: KProperty1<VS, T2>,
  property3: KProperty1<VS, T3>,
  property4: KProperty1<VS, T4>,
  property5: KProperty1<VS, T5>,
  property6: KProperty1<VS, T6>,
  property7: KProperty1<VS, T7>,
  property8: KProperty1<VS, T8>,
) = Tuple8(
  property1.get(this),
  property2.get(this),
  property3.get(this),
  property4.get(this),
  property5.get(this),
  property6.get(this),
  property7.get(this),
  property8.get(this),
)

@Composable
public fun <VS : Any, E, T1, T2, T3, T4, T5, T6, T7, T8, T9> LoomViewModel<VS, E>.collectAsState(
  property1: KProperty1<VS, T1>,
  property2: KProperty1<VS, T2>,
  property3: KProperty1<VS, T3>,
  property4: KProperty1<VS, T4>,
  property5: KProperty1<VS, T5>,
  property6: KProperty1<VS, T6>,
  property7: KProperty1<VS, T7>,
  property8: KProperty1<VS, T8>,
  property9: KProperty1<VS, T9>,
): State<Tuple9<T1, T2, T3, T4, T5, T6, T7, T8, T9>> =
  collectAsStateWithMapper {
    it.toTuple9(
      property1,
      property2,
      property3,
      property4,
      property5,
      property6,
      property7,
      property8,
      property9
    )
  }

private fun <VS : Any, T1, T2, T3, T4, T5, T6, T7, T8, T9> VS.toTuple9(
  property1: KProperty1<VS, T1>,
  property2: KProperty1<VS, T2>,
  property3: KProperty1<VS, T3>,
  property4: KProperty1<VS, T4>,
  property5: KProperty1<VS, T5>,
  property6: KProperty1<VS, T6>,
  property7: KProperty1<VS, T7>,
  property8: KProperty1<VS, T8>,
  property9: KProperty1<VS, T9>,
) = Tuple9(
  property1.get(this),
  property2.get(this),
  property3.get(this),
  property4.get(this),
  property5.get(this),
  property6.get(this),
  property7.get(this),
  property8.get(this),
  property9.get(this),
)

@Composable
public fun <VS : Any, E, T1, T2, T3, T4, T5, T6, T7, T8, T9, T10> LoomViewModel<VS, E>.collectAsState(
  property1: KProperty1<VS, T1>,
  property2: KProperty1<VS, T2>,
  property3: KProperty1<VS, T3>,
  property4: KProperty1<VS, T4>,
  property5: KProperty1<VS, T5>,
  property6: KProperty1<VS, T6>,
  property7: KProperty1<VS, T7>,
  property8: KProperty1<VS, T8>,
  property9: KProperty1<VS, T9>,
  property10: KProperty1<VS, T10>,
): State<Tuple10<T1, T2, T3, T4, T5, T6, T7, T8, T9, T10>> =
  collectAsStateWithMapper {
    it.toTuple10(
      property1,
      property2,
      property3,
      property4,
      property5,
      property6,
      property7,
      property8,
      property9,
      property10
    )
  }

private fun <VS : Any, T1, T2, T3, T4, T5, T6, T7, T8, T9, T10> VS.toTuple10(
  property1: KProperty1<VS, T1>,
  property2: KProperty1<VS, T2>,
  property3: KProperty1<VS, T3>,
  property4: KProperty1<VS, T4>,
  property5: KProperty1<VS, T5>,
  property6: KProperty1<VS, T6>,
  property7: KProperty1<VS, T7>,
  property8: KProperty1<VS, T8>,
  property9: KProperty1<VS, T9>,
  property10: KProperty1<VS, T10>,
) = Tuple10(
  property1.get(this),
  property2.get(this),
  property3.get(this),
  property4.get(this),
  property5.get(this),
  property6.get(this),
  property7.get(this),
  property8.get(this),
  property9.get(this),
  property10.get(this),
)
