package io.naraplatform.daysman.mediator.spec.domain.transport;

import io.naradrama.prologue.domain.cqrs.*;
import io.naradrama.prologue.domain.cqrs.broker.AbstractStreamEventMessage;
import io.naradrama.prologue.domain.cqrs.broker.StreamEventMessage;
import io.naradrama.prologue.domain.cqrs.command.CommandResponse;
import io.naradrama.prologue.domain.cqrs.command.CqrsCommand;
import io.naradrama.prologue.domain.cqrs.command.CqrsCommandType;
import io.naradrama.prologue.domain.cqrs.event.CqrsEvent;
import io.naradrama.prologue.domain.cqrs.event.CqrsEventType;
import io.naradrama.prologue.domain.cqrs.query.CqrsQuery;
import io.naradrama.prologue.domain.cqrs.query.CqrsQueryType;
import io.naradrama.prologue.util.json.JsonSerializable;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;

import java.sql.Timestamp;
import java.util.List;

@Getter
@Setter
@NoArgsConstructor
public class EventMessage implements JsonSerializable {
    // From AbstractStreamEventMessage
    private DomainMessageType domainMessageType;    // Equals with DomainMessage.domainMessageType
    private String id;                              // Equals with DomainMessage.id
    private long time;                              // Equals with AbstractStreamEventMessage.timestamp

    // From StreamEventMessage
    private String payloadClass;
    private String payload;
    private String serviceName;

    // From TraceHeader
    private String traceId;             // uuid
    private String userId;              // actorId or stationPlayerId, etc.
    private String parentService;       // caller(parent) service name. ex, io.naradrama.timecard
    private String service;             // callee service name. ex, io.naradrama.depot
    private String parentMessage;       // caller DomainMessage name. ex, LeaveCommand
    private String message;             // callee DomainMessage name. ex, LeaveEvent
    private long traceTime;

    // From CqrsEvent
    private CqrsEventType cqrsEventType;
    private String commandId;                       // Equals with CqrsCommand.id
    private String producedTimestamp;               // Equals with CqrsCommand.requestTime or CqrsQuery.requestTime

    // From CqrsCommand or CqrsQuery
    private CqrsCommandType cqrsCommandType;
    private CqrsQueryType cqrsQueryType;
    private long waitingTime;
    private boolean requestFailed;

    // From FailureMessage
    private String exceptionName;
    private String exceptionMessage;
    private String exceptionCode;

    // From CommandResponse
    private List<String> responseEntityIds;
    private boolean responseResult;

    private String baseSentId;
    // From Spring transaction listener
    private boolean failed;

    public EventMessage(StreamEventMessage message, String baseSentId) {
        //
        this(message, baseSentId, false);
    }

    public EventMessage(StreamEventMessage message, String baseSentId, boolean failed) {
        //
        fillByAbstractStreamEventMessage(message);
        fillByStreamEventMessage(message);
        fillByPayloadObject(message.payloadObject());

        this.baseSentId = baseSentId;
        this.failed = failed;
    }

    private void fillByPayloadObject(Object payloadObject) {
        //
        if (payloadObject instanceof CqrsEvent) {
            fillByCqrsEvent((CqrsEvent) payloadObject);
        } else if (payloadObject instanceof CqrsCommand) {
            fillByCqrsCommand((CqrsCommand) payloadObject);
        } else if (payloadObject instanceof CqrsQuery) {
            fillByCqrsQuery((CqrsQuery<?>) payloadObject);
        }
    }

    private void fillByCqrsEvent(CqrsEvent cqrsEvent) {
        //
        fillByTraceHeader(cqrsEvent.getTraceHeader());
        this.cqrsEventType = cqrsEvent.getCqrsEventType();
        this.commandId = cqrsEvent.getCommandId();
        this.producedTimestamp = cqrsEvent.getProducedTimestamp();
    }

    private void fillByCqrsCommand(CqrsCommand cqrsCommand) {
        //
        fillByTraceHeader(cqrsCommand.getTraceHeader());
        this.commandId = cqrsCommand.getId();
        this.cqrsCommandType = cqrsCommand.getCqrsCommandType();
        this.producedTimestamp = cqrsCommand.genRequestTimestamp();
        this.waitingTime = cqrsCommand.getWaitingTime();
        this.requestFailed = cqrsCommand.isRequestFailed();
        fillByFailureMessage(cqrsCommand.getFailureMessage());
        fillByCommandResponse(cqrsCommand.getCommandResponse());
    }

    private void fillByCqrsQuery(CqrsQuery<?> cqrsQuery) {
        //
        this.cqrsQueryType = cqrsQuery.getCqrsQueryType();
        this.producedTimestamp = cqrsQuery.genRequestTimestamp();
        this.waitingTime = cqrsQuery.getWaitingTime();
        this.requestFailed = cqrsQuery.isRequestFailed();
        fillByFailureMessage(cqrsQuery.getFailureMessage());
    }

    private void fillByAbstractStreamEventMessage(AbstractStreamEventMessage message) {
        //
        this.domainMessageType = message.getDomainMessageType();
        this.id = message.getId();
        this.time = Timestamp.valueOf(message.getTimestamp()).getTime();
    }

    private void fillByStreamEventMessage(StreamEventMessage message) {
        //
        this.payloadClass = message.getPayloadClass();
        this.payload = message.getPayload();
        this.serviceName = message.getServiceName();
    }

    private void fillByTraceHeader(TraceHeader traceHeader) {
        //
        if (traceHeader == null) {
            return;
        }
        this.traceId = traceHeader.getTraceId();
        this.userId = traceHeader.getUserId();
        this.parentService = traceHeader.getParentService();
        this.service = traceHeader.getService();
        this.parentMessage = traceHeader.getParentMessage();
        this.message = traceHeader.getMessage();
        this.traceTime = traceHeader.getTime();
    }

    private void fillByFailureMessage(FailureMessage failureMessage) {
        //
        if (failureMessage == null) {
            return;
        }
        this.exceptionName = failureMessage.getExceptionName();
        this.exceptionMessage = failureMessage.getExceptionMessage();
        this.exceptionCode = failureMessage.getExceptionCode();
    }

    private void fillByCommandResponse(CommandResponse commandResponse) {
        //
        if (commandResponse == null) {
            return;
        }
        this.responseEntityIds = commandResponse.getEntityIds();
        this.responseResult = commandResponse.isResult();
    }

    public StreamEventMessage toStreamEventMessage() {
        //
        StreamEventMessage event = new StreamEventMessage();
        event.setId(this.id);
        event.setDomainMessageType(this.domainMessageType);
        event.setPayloadClass(this.payloadClass);
        event.setPayload(this.payload);
        event.setTimestamp(new Timestamp(this.time).toString());
        event.setServiceName(this.serviceName);
        return event;
    }

}
