package io.mitter.sdk.java.clients

import com.fasterxml.jackson.databind.JsonNode
import feign.Headers
import feign.Param
import feign.QueryMap
import feign.RequestLine
import io.mitter.data.domain.annotations.Identifiable
import io.mitter.data.domain.user.User
import io.mitter.models.mardle.messaging.*
import io.mitter.models.mardle.wire.ChannelTimelineEvent
import io.mitter.models.mardle.wire.Paginated
import io.mitter.models.mardle.wire.PatchChannelParticipation
import io.mitter.named.resources.http.endpoints.ChannelEndpointsV1
import io.mitter.named.resources.http.endpoints.ChannelMessageEndpointsV1
import io.mitter.named.resources.http.endpoints.UserEndpointsV1
import io.mitter.sdk.java.support.IdentifiableExpander
import io.mitter.sdk.java.utils.StandardHeaderLines
import io.mitter.spi.java.clients.MitterChannelsClient

/**
 *
 * @author Rohan Prabhu (rohan@rohanprabhu.com)
 */
@Headers(StandardHeaderLines.AcceptApplicationJson)
interface HttpMitterChannelsClient : MitterChannelsClient {

    @RequestLine("POST " + ChannelEndpointsV1.Base)
    @Headers(StandardHeaderLines.ContentTypeApplicationJson)
    override fun newChannel(channel: Channel): Identifiable<Channel>

    @RequestLine("DELETE " + ChannelEndpointsV1.Specified.ForChannelId)
    override fun deleteChannel(
        @Param(ChannelEndpointsV1.ChannelIdParam, expander = IdentifiableExpander::class)
        channelId: Identifiable<Channel>
    )

    @RequestLine("GET " + ChannelEndpointsV1.Specified.ForChannelIds)
    override fun getChannels(
        @Param(ChannelEndpointsV1.ChannelIdsParam, expander = IdentifiableExpander::class)
        channelIds: List<Identifiable<Channel>>
    ): List<Channel>

    @RequestLine("GET " + ChannelEndpointsV1.Base)
    override fun getChannels( @QueryMap queryMap: Map<String, Any> ): Paginated<Channel>

    @RequestLine("DELETE " + ChannelMessageEndpointsV1.Base)
    override fun deleteMessagesFromChannel(
        @Param(ChannelEndpointsV1.ChannelIdParam, expander = IdentifiableExpander::class)
        channelId: Identifiable<Channel>
    )

    /* ~~ Channel Timeline event operations ~~ */

    @RequestLine("POST " + ChannelEndpointsV1.TimelineEvents.ForChannelId)
    @Headers(StandardHeaderLines.ContentTypeApplicationJson)
    override fun addChannelTimelineEvents(
        @Param(ChannelEndpointsV1.ChannelIdParam, expander = IdentifiableExpander::class)
        channelId: Identifiable<Channel>,
        timelineEvent: TimelineEvent
    )

    @RequestLine("GET " + ChannelEndpointsV1.TimelineEvents.ForChannelIds)
    override fun getChannelTimelineEvents(
        @Param(ChannelEndpointsV1.ChannelIdsParam, expander = IdentifiableExpander::class)
        channelIds: Identifiable<Channel>
    ): List<ChannelTimelineEvent>

    /* ~~ Channel streams operations ~~ */

    @RequestLine("POST " + ChannelEndpointsV1.Streams.Specified.ByStreamId)
    @Headers(StandardHeaderLines.ContentTypeApplicationJson)
    override fun sendStreamData(
        @Param(ChannelEndpointsV1.ChannelIdParam, expander = IdentifiableExpander::class)
        channelId: Identifiable<Channel>,
        @Param(ChannelEndpointsV1.Streams.StreamIdParam, expander = IdentifiableExpander::class)
        streamId: Identifiable<Stream>,
        streamData: ContextFreeMessage
    )

    @RequestLine("GET " + ChannelEndpointsV1.Streams.Specified.ByStreamId)
    override fun getChannelStreamByStreamId(
        @Param(ChannelEndpointsV1.ChannelIdParam, expander = IdentifiableExpander::class)
        channelId: Identifiable<Channel>,
        @Param(ChannelEndpointsV1.Streams.StreamIdParam, expander = IdentifiableExpander::class)
        streamId: Identifiable<Stream>
    ) : Stream

    @RequestLine("POST " + ChannelEndpointsV1.Streams.ForChannelId)
    override fun createChannelStream(
        @Param(ChannelEndpointsV1.ChannelIdParam, expander = IdentifiableExpander::class)
        channelId: Identifiable<Channel>,
        channel: Channel
    ) : Stream

    /* ~~ Participant operations -- */
    @RequestLine("POST " + ChannelEndpointsV1.Participation.ForParticipants)
    @Headers(StandardHeaderLines.ContentTypeApplicationJson)
    override fun addParticipantToChannel(
        @Param(ChannelEndpointsV1.ChannelIdParam, expander = IdentifiableExpander::class)
        channelId: Identifiable<Channel>,
        channelParticipation: ChannelParticipation
    )

    @RequestLine("GET " + ChannelEndpointsV1.Participation.ForParticipants)
    override fun getChannelParticipations(
        @Param(ChannelEndpointsV1.ChannelIdParam, expander = IdentifiableExpander::class)
        channelId: Identifiable<Channel>
    ): List<ChannelParticipation>

    @RequestLine("DELETE " + ChannelEndpointsV1.Participation.Specified.ForParticipantId)
    override fun deleteParticipantFromChannel(
        @Param(ChannelEndpointsV1.ChannelIdParam, expander = IdentifiableExpander::class)
        channelId: Identifiable<Channel>,
        @Param(ChannelEndpointsV1.Participation.ParticipantIdParam, expander = IdentifiableExpander::class)
        participantId: Identifiable<User>
    )

    @RequestLine("GET " + UserEndpointsV1.Channels.ForUserId)
    override fun getChannelsWithParticipant(
        @Param(UserEndpointsV1.UserIdParam, expander = IdentifiableExpander::class)
        userId: Identifiable<User>
    ): List<ParticipatedChannel>

    @RequestLine("PATCH " + ChannelEndpointsV1.Participation.Specified.ForParticipantId)
    @Headers(StandardHeaderLines.ContentTypeApplicationJson)
    override fun updateParticipationStatus(
        @Param(ChannelEndpointsV1.ChannelIdParam, expander = IdentifiableExpander::class)
        channelId: Identifiable<Channel>,
        @Param(ChannelEndpointsV1.Participation.ParticipantIdParam, expander = IdentifiableExpander::class)
        participantId: Identifiable<User>,
        patchChannelParticipation: PatchChannelParticipation
    )

    /* Channel Metadata Operations  */

    @RequestLine("GET " + ChannelEndpointsV1.Base)
    override fun getChannelsWithMetadata( @QueryMap queryMap: Map<String, Any> ): List<Channel>

    @RequestLine("POST " + ChannelEndpointsV1.Metadata.ForChannelId)
    @Headers(StandardHeaderLines.ContentTypeApplicationJson)
    override fun addChannelMetadata(
        @Param(ChannelEndpointsV1.ChannelIdParam, expander = IdentifiableExpander::class)
        channelId: Identifiable<Channel>,
        metadata: JsonNode
    )
}
