package io.mitter.sdk.java

import com.fasterxml.jackson.databind.ObjectMapper
import feign.Feign
import feign.Logger
import feign.RequestInterceptor
import feign.jackson.JacksonDecoder
import feign.jackson.JacksonEncoder
import feign.okhttp.OkHttpClient
import io.mitter.sdk.java.clients.*
import io.mitter.sdk.java.support.MitterSdkSupport
import io.mitter.sdk.java.support.encoder.MitterSupplementalEncoder
import io.mitter.sdk.java.support.interceptors.*
import io.mitter.spi.java.MitterCentralBaseClientFactory
import io.mitter.spi.java.clients.MitterUsersClient
import io.mitter.spi.java.clients.hlc.MitterMessagesHlcClient
import io.mitter.spi.java.credentials.*
import mu.KotlinLogging
import java.util.*

/**
 *
 * @author Rohan Prabhu (rohan@rohanprabhu.com)
 */
class MitterCentralClientFactory : MitterCentralBaseClientFactory {
    companion object {
        const val MitterCentralApiEndpoint = "https://api.mitter.io"
        const val LoggerStatementLimit = 600
    }

    private val objectMapper   : ObjectMapper = MitterSdkSupport.configureJackson(ObjectMapper())
    private val jacksonDecoder : JacksonDecoder
    private val jacksonEncoder : JacksonEncoder
    override val mitterAccessCredentials: MitterAccessCredentials?
    private val mitterCentralEndpoint : String?
    override var enableLogging : Boolean = false

    constructor(mitterAccessCredentials: MitterAccessCredentials) {
        this.mitterAccessCredentials = mitterAccessCredentials
        this.mitterCentralEndpoint = null
    }

    constructor(mitterCentralEndpoint: String) {
        this.mitterAccessCredentials = null
        this.mitterCentralEndpoint = mitterCentralEndpoint
    }

    constructor(mitterAccessCredentials: MitterAccessCredentials, mitterCentralEndpoint: String) {
        this.mitterAccessCredentials = mitterAccessCredentials
        this.mitterCentralEndpoint = mitterCentralEndpoint
    }

    constructor() {
        this.mitterCentralEndpoint = null
        this.mitterAccessCredentials = null
    }

    init {
        this.jacksonDecoder = JacksonDecoder(this.objectMapper)
        this.jacksonEncoder = JacksonEncoder(this.objectMapper)
    }

    fun setLoggingEnabled(enable: Boolean) { this.enableLogging = enable }

    /*
      -- CLIENT METHODS --
     */

    /* ****
       Access Key Test Client Methods
       ****
     */
    override fun accessKeyTestClient(mitterAccessCredentials: MitterAccessCredentials?) =
        accessKeyTestClient(mitterAccessCredentials, mitterCentralEndpoint ?: MitterCentralApiEndpoint)

    fun accessKeyTestClient(
            mitterAccessCredentials: MitterAccessCredentials?, centralApiEndpoint: String,
            enableFullLogging: Boolean = this.enableLogging
    ) : HttpMitterAccessKeyTestClient {
        return getBaseBuilder(mitterAccessCredentials, enableFullLogging, HttpMitterAccessKeyTestClient::class.java)
            .target(HttpMitterAccessKeyTestClient::class.java, centralApiEndpoint)
    }

    /* ****
       ACL Operations Client Methods
       ****
     */

    override fun aclOperationsClient(mitterAccessCredentials: MitterAccessCredentials?) =
        aclOperationsClient(mitterAccessCredentials, mitterCentralEndpoint ?: MitterCentralApiEndpoint)

    fun aclOperationsClient(
            mitterAccessCredentials: MitterAccessCredentials?, centralApiEndpoint: String,
            enableFullLogging: Boolean = this.enableLogging
    ) : HttpMitterAclOperationsClient {
        return getBaseBuilder(mitterAccessCredentials, enableFullLogging, HttpMitterAccessKeyTestClient::class.java)
            .target(HttpMitterAclOperationsClient::class.java, centralApiEndpoint)
    }

    /* ****
       Application properties Client Methods
       ****
     */

    override fun applicationPropertiesClient(mitterAccessCredentials: MitterAccessCredentials?) =
        applicationPropertiesClient(mitterAccessCredentials, mitterCentralEndpoint ?: MitterCentralApiEndpoint)

    fun applicationPropertiesClient(
            mitterAccessCredentials: MitterAccessCredentials?, centralApiEndpoint: String,
            enableFullLogging: Boolean = this.enableLogging
    ) : HttpMitterApplicationPropertiesClient {
        return getBaseBuilder(mitterAccessCredentials, enableFullLogging, HttpMitterApplicationClient::class.java)
            .target(HttpMitterApplicationPropertiesClient::class.java, centralApiEndpoint)
    }

    /* ****
       Application AccessKey Client Methods
       ****
     */

    override fun applicationAccessKeyClient(mitterAccessCredentials: MitterAccessCredentials?) =
        applicationAccessKeyClient(mitterAccessCredentials, mitterCentralEndpoint ?: MitterCentralApiEndpoint)

    fun applicationAccessKeyClient(
            mitterAccessCredentials: MitterAccessCredentials?,
            centralApiEndpoint: String,
            enableFullLogging: Boolean = this.enableLogging
    ) : HttpMitterApplicationAccessKeyClient {
        return getBaseBuilder(mitterAccessCredentials,
            enableFullLogging,
            HttpMitterApplicationAccessKeyClient::class.java)
            .target(HttpMitterApplicationAccessKeyClient::class.java, centralApiEndpoint)
    }

    /* ****
        Application Client Methods
    ****/

    override fun applicationClient(mitterAccessCredentials: MitterAccessCredentials?) =
        applicationClient(mitterAccessCredentials, mitterCentralEndpoint ?: MitterCentralApiEndpoint)

    fun applicationClient(
            mitterAccessCredentials: MitterAccessCredentials?, centralApiEndpoint: String,
            enableFullLogging: Boolean = this.enableLogging
    ) : HttpMitterApplicationClient {
        return getBaseBuilder(mitterAccessCredentials, enableFullLogging, HttpMitterApplicationClient::class.java)
            .target(HttpMitterApplicationClient::class.java, centralApiEndpoint)
    }

    /* ****
      Subscriber Client Methods
      ****
    */

    override fun subscriberClient(mitterAccessCredentials: MitterAccessCredentials?) =
        subscriberClient(mitterAccessCredentials, mitterCentralEndpoint ?: MitterCentralApiEndpoint)

    fun subscriberClient(
            mitterAccessCredentials: MitterAccessCredentials?, centralApiEndpoint: String,
            enableFullLogging: Boolean = this.enableLogging
    ) : HttpMitterAuthenticationEndpointsClient {
        return getBaseBuilder(mitterAccessCredentials, enableFullLogging, HttpMitterAuthenticationEndpointsClient::class.java)
            .target(HttpMitterAuthenticationEndpointsClient::class.java, centralApiEndpoint)
    }

    /* ****
       Subscriber Client Methods
       ****
    */

    override fun subscriberAccessApiClient(mitterAccessCredentials: MitterAccessCredentials?) =
        subscriberAccessApiClient(mitterAccessCredentials, mitterCentralEndpoint ?: MitterCentralApiEndpoint)

    fun subscriberAccessApiClient(
            mitterAccessCredentials: MitterAccessCredentials?, centralApiEndpoint: String,
            enableFullLogging: Boolean = this.enableLogging
    ) : HttpMitterSubscriberAccessApiClient {
        return getBaseBuilder(mitterAccessCredentials, enableFullLogging, HttpMitterSubscriberAccessApiClient::class.java)
            .target(HttpMitterSubscriberAccessApiClient::class.java, centralApiEndpoint)
    }

    /* ****
       Users Client Methods
       ****
     */

    override fun usersClient(mitterAccessCredentials: MitterAccessCredentials?) =
        usersClient(mitterAccessCredentials, mitterCentralEndpoint ?: MitterCentralApiEndpoint)

    fun usersClient(
            mitterAccessCredentials: MitterAccessCredentials?, centralApiEndpoint: String,
            enableFullLogging: Boolean = this.enableLogging
    ) : MitterUsersClient {
        return getBaseBuilder(mitterAccessCredentials, enableFullLogging, HttpMitterUsersClient::class.java)
            .requestInterceptor(MetadataEncodingInterceptor())
            .target(HttpMitterUsersClient::class.java, centralApiEndpoint)
    }

    /* ****
       User Auth Client Methods
       ****
     */

    override fun userAuthClient(mitterAccessCredentials: MitterAccessCredentials?) =
        userAuthClient(mitterAccessCredentials, mitterCentralEndpoint ?: MitterCentralApiEndpoint)

    fun userAuthClient(mitterAccessCredentials: MitterAccessCredentials?, centralApiEndpoint: String,
                       enableFullLogging: Boolean = this.enableLogging) : HttpMitterUsersAuthClient {
        return getBaseBuilder(mitterAccessCredentials, enableFullLogging, HttpMitterUsersAuthClient::class.java)
            .target(HttpMitterUsersAuthClient::class.java, centralApiEndpoint)
    }

    /* ****
      User Presence Client Methods
      ****
    */

    override fun userPresenceClient(mitterAccessCredentials: MitterAccessCredentials?) =
        userPresenceClient(mitterAccessCredentials, mitterCentralEndpoint ?: MitterCentralApiEndpoint)

    fun userPresenceClient(mitterAccessCredentials: MitterAccessCredentials?, centralApiEndpoint: String,
                           enableFullLogging: Boolean = this.enableLogging) : HttpMitterUserPresenceClient {
        return getBaseBuilder(mitterAccessCredentials, enableFullLogging, HttpMitterUserPresenceClient::class.java)
            .target(HttpMitterUserPresenceClient::class.java, centralApiEndpoint)
    }

    /* ****
      Entity Profile Client Methods
      ****
    */

    override fun entityProfileClient(mitterAccessCredentials: MitterAccessCredentials?) =
        entityProfileClient(mitterAccessCredentials, mitterCentralEndpoint ?: MitterCentralApiEndpoint)

    fun entityProfileClient(mitterAccessCredentials: MitterAccessCredentials?, centralApiEndpoint: String,
                            enableFullLogging: Boolean = this.enableLogging) : HttpMitterEntityProfileClient {
        return getBaseBuilder(mitterAccessCredentials, enableFullLogging, HttpMitterEntityProfileClient::class.java)
            .target(HttpMitterEntityProfileClient::class.java, centralApiEndpoint)
    }

    /* ****
       Channels Client Methods
       ****
     */

    override fun channelsClient(mitterAccessCredentials: MitterAccessCredentials?) =
        channelsClient(mitterAccessCredentials, mitterCentralEndpoint ?: MitterCentralApiEndpoint)

    fun channelsClient(
            mitterAccessCredentials: MitterAccessCredentials?, centralApiEndpoint: String,
            enableFullLogging: Boolean = this.enableLogging
    ) : HttpMitterChannelsClient {
        return getBaseBuilder(mitterAccessCredentials, enableFullLogging, HttpMitterChannelsClient::class.java)
            .requestInterceptor(MetadataEncodingInterceptor())
            .target(HttpMitterChannelsClient::class.java, centralApiEndpoint)
    }

    /* ****
       Delivery Endpoint Client Methods
       ****
     */

    override fun deliveryEndpointClient(mitterAccessCredentials: MitterAccessCredentials?) =
        deliveryEndpointClient(mitterAccessCredentials, mitterCentralEndpoint ?: MitterCentralApiEndpoint)

    fun deliveryEndpointClient(
            mitterAccessCredentials: MitterAccessCredentials?, centralApiEndpoint: String,
            enableFullLogging: Boolean = this.enableLogging
    ) : HttpMitterDeliveryEndpointsClient {
        return getBaseBuilder(mitterAccessCredentials, enableFullLogging, HttpMitterDeliveryEndpointsClient::class.java)
            .target(HttpMitterDeliveryEndpointsClient::class.java, centralApiEndpoint)
    }

    /* ****
       Messages Client Methods
       ****
     */

    override fun messagesClient(mitterAccessCredentials: MitterAccessCredentials?) =
        messagesClient(mitterAccessCredentials, mitterCentralEndpoint ?: MitterCentralApiEndpoint)

    fun messagesClient(
            mitterAccessCredentials: MitterAccessCredentials?, centralApiEndpoint: String,
            enableFullLogging: Boolean = this.enableLogging
    ) : HttpMitterMessagesClient {
        return getBaseBuilder(mitterAccessCredentials, enableFullLogging, HttpMitterMessagesClient::class.java)
            .target(HttpMitterMessagesClient::class.java, centralApiEndpoint)
    }

    /* ****
       Counts Client Methods
       ****
     */

    override fun countsClient(mitterAccessCredentials: MitterAccessCredentials?) =
        countsClient(mitterAccessCredentials, mitterCentralEndpoint ?: MitterCentralApiEndpoint)

    fun countsClient(
        mitterAccessCredentials: MitterAccessCredentials?, centralApiEndpoint: String,
        enableFullLogging: Boolean = this.enableLogging
    ) : HttpMitterCountsClient {
        return getBaseBuilder(mitterAccessCredentials, enableFullLogging, HttpMitterCountsClient::class.java)
            .target(HttpMitterCountsClient::class.java, centralApiEndpoint)
    }

    /* ****
       Messages HLC Client methods
       ****
     */

    override fun messagesHlcClient(
        mitterAccessCredentials: MitterAccessCredentials?
    ) = MitterMessagesHlcClient(
        messagesClient(mitterAccessCredentials)
    )

    fun messagesHlcClient(
            mitterAccessCredentials: MitterAccessCredentials?, centralApiEndpoint: String,
            enableFullLogging: Boolean
    ) = MitterMessagesHlcClient(
        messagesClient(mitterAccessCredentials, centralApiEndpoint, enableFullLogging)
    )

    /*
      -- END OF CLIENT METHODS --
     */

    private fun getBaseBuilder(
            mitterAccessCredentials: MitterAccessCredentials?, enableFullLogging: Boolean,
            target: Class<*>
    ) : Feign.Builder =
        mitterAccessCredentials?.let {
            Feign.builder()
                .decoder(this.jacksonDecoder)
                .encoder(MitterSupplementalEncoder(this.jacksonEncoder))
                .let {
                    if (enableFullLogging) {
                        it.logLevel(Logger.Level.FULL).logger(MitterLogger(target))
                    } else {
                        it
                    }
                }
                .client(OkHttpClient())
                .configureInterceptors(it)
        } ?: throw IllegalStateException("Must set the credentials either in the factory," +
            " or pass it to the individual client creator method")

    private fun Feign.Builder.configureInterceptors(mitterAccessCredentials: MitterAccessCredentials) : Feign.Builder {
        val interceptorChain = mutableListOf<RequestInterceptor>()

        fun resolveInterceptor(mitterAccessCredentials: MitterAccessCredentials) {
            when (mitterAccessCredentials) {
                is MitterApplicationCredentials ->
                    when (mitterAccessCredentials) {
                        is MitterApplicationAccessKeyCredentials ->
                            interceptorChain.add(MitterApplicationAccessKeySigner(mitterAccessCredentials))

                        is MitterSudoApplicationCredentials -> {
                            interceptorChain.add(MitterSudoApplicationSetter(mitterAccessCredentials))
                            resolveInterceptor(mitterAccessCredentials.mitterSubscriberCredentials)
                        }
                    }

                is MitterUserCredentials ->
                    when (mitterAccessCredentials) {
                        is MitterUserTokenCredentials ->
                            interceptorChain.add(MitterUserTokenCredentialSetter(mitterAccessCredentials))

                        is MitterSudoUserCredentials -> {
                            interceptorChain.add(MitterApplicationSudoUserSetter(mitterAccessCredentials))
                            resolveInterceptor(mitterAccessCredentials.mitterApplicationCredentials)
                        }
                    }

                is MitterSubscriberCredentials -> {
                    when(mitterAccessCredentials) {
                        is MitterSubscriberApiAccessCredentials -> {
                            interceptorChain.add(MitterSubscriberAccessKeySigner(mitterAccessCredentials))
                        }

                        is MitterAuthorizeSubscriberCredentials -> {
                            interceptorChain.add(MitterSubscriberAuthorizationSetter(mitterAccessCredentials))
                        }
                    }
                }

                is MitterAnonymousCredentials -> {
                   // do nothing
                   // this is a call that is made without any authorization information
                }
            }
        }

        resolveInterceptor(mitterAccessCredentials)
        return this.requestInterceptor(MitterChainedRequestInterceptor(interceptorChain))
    }

    private fun getApiEndpoint(apiEndpoint: String?) : String =
        apiEndpoint ?: MitterCentralApiEndpoint

    private class MitterLogger(loggingClass: Class<*>) : Logger() {
        private val logger = KotlinLogging.logger("[WIRE] " + loggingClass.simpleName)

        override fun log(configKey: String?, format: String?, vararg args: Any?) {
            logger.info { Formatter().format(format, *args).toString()
                .let {
                    if (it.length > LoggerStatementLimit) {
                        it.substring(0, LoggerStatementLimit) + " .. [TRUNCATED to $LoggerStatementLimit characters]"
                    } else {
                        it
                    }
                }
            }
        }
    }
}
